import { faker } from '@faker-js/faker';
import { STATUS_OPTIONS, BRANCH_OPTIONS, MEPS_LOCATIONS } from './constants';

export interface VerificationData {
  status: string;
  branch: string;
  firstName: string;
  lastName: string;
  dob: Date | null;
  startDate: Date | null; 
  dischargeDate: Date | null;
  email: string;
  recruitmentCenter: string;
  rank?: string;
  notes?: string;
  logo?: string;
  photo?: string;
  authorizedOfficialName?: string;
}

export function generateProfile(): VerificationData {
  // 1. Status & Branch
  const status = faker.helpers.arrayElement(STATUS_OPTIONS);
  const branch = faker.helpers.arrayElement(BRANCH_OPTIONS);

  // 2. Name
  const firstName = faker.person.firstName();
  const lastName = faker.person.lastName();

  // 3. Dates
  // Age 20-55
  const dob = faker.date.birthdate({ min: 20, max: 55, mode: 'age' });

  // Enlistment age 18-27
  const enlistmentAge = faker.number.int({ min: 18, max: 27 });
  const enlistmentDate = new Date(dob);
  enlistmentDate.setFullYear(enlistmentDate.getFullYear() + enlistmentAge);
  enlistmentDate.setMonth(faker.number.int({ min: 0, max: 11 }));
  enlistmentDate.setDate(faker.number.int({ min: 1, max: 28 }));

  // Determine Discharge Date based on Status to ensure logical consistency
  let dischargeDate = new Date();
  
  if (status === 'Active Duty' || status === 'Reserve' || status === 'National Guard') {
      // Active: Discharge date is likely in the future (Time left on contract)
      // Generates a date 1 month to 4 years in the future
      // Bias towards "Transitioning" (< 12 months) for testing visibility
      const daysFuture = faker.helpers.arrayElement([
          faker.number.int({ min: 30, max: 300 }), // Transitioning
          faker.number.int({ min: 400, max: 1500 }) // Stable
      ]);
      dischargeDate = new Date();
      dischargeDate.setDate(dischargeDate.getDate() + daysFuture);
  } else {
      // Veteran / Retired: Discharge date is in the past
      // Bias towards "Recently Separated" (< 12 months) for testing visibility
      const daysPast = faker.helpers.arrayElement([
          faker.number.int({ min: 10, max: 300 }), // Recently Separated
          faker.number.int({ min: 400, max: 5000 }) // Long time ago
      ]);
      dischargeDate = new Date();
      dischargeDate.setDate(dischargeDate.getDate() - daysPast);
  }

  // 4. Email
  // firstname.lastname####@domain.com
  const randomNum = faker.number.int({ min: 1000, max: 9999 });
  const domain = faker.helpers.arrayElement(['gmail.com', 'outlook.com', 'proton.me', 'yahoo.com']);
  const email = `${firstName.toLowerCase()}.${lastName.toLowerCase()}${randomNum}@${domain}`;

  // 5. Recruitment Center (MEPS)
  const recruitmentCenter = faker.helpers.arrayElement(MEPS_LOCATIONS) as string;

  // 6. Authorized Official
  const officialRank = faker.helpers.arrayElement(['CPT', 'MAJ', 'LTC', 'COL', 'GEN']);
  const officialName = `${officialRank} ${faker.person.firstName().charAt(0)}. ${faker.person.lastName()}`;

  return {
    status,
    branch,
    firstName,
    lastName,
    dob,
    startDate: enlistmentDate,
    dischargeDate,
    recruitmentCenter,
    email,
    authorizedOfficialName: officialName,
  };
}
