import { Pool, PoolConfig, QueryResult, QueryResultRow } from 'pg';

const poolConfig: PoolConfig = {
  connectionString: process.env.DATABASE_URL,
  // Fallback to individual vars if DATABASE_URL is not set or valid
  host: process.env.PGHOST || 'localhost',
  port: parseInt(process.env.PGPORT || '5432'),
  database: process.env.PGDATABASE || 'genekei',
  user: process.env.PGUSER || 'genekei_user',
  password: process.env.PGPASSWORD,
  max: 10, // Max number of clients in the pool
  idleTimeoutMillis: 30000,
};

// Use a global variable to store the pool instance in development
// This prevents creating multiple pools during hot reloads
const globalForDb = globalThis as unknown as {
  conn: Pool | undefined;
};

const pool = globalForDb.conn ?? new Pool(poolConfig);

if (process.env.NODE_ENV !== 'production') {
  globalForDb.conn = pool;
}

pool.on('error', (err) => {
  console.error('Unexpected error on idle client', err);
  process.exit(-1);
});

export const db = {
  query: async <R extends QueryResultRow = any, I extends any[] = any[]>(
    text: string,
    params?: I
  ): Promise<QueryResult<R>> => {
    const start = Date.now();
    try {
      const res = await pool.query<R, I>(text, params);
      const duration = Date.now() - start;
      // console.log('executed query', { text, duration, rows: res.rowCount });
      return res;
    } catch (error) {
        console.error('Error executing query', { text, error });
        throw error;
    }
  },
  
  // Helper to get a client for transactions
  getClient: () => pool.connect(),
};
