"use client";

import { useState, useEffect } from 'react';
import { Template, TemplateType, STORAGE_KEY_TEMPLATES } from '@/types/templates';

const DEFAULT_TEMPLATES: Template[] = [
    {
        id: 'default-harvard',
        name: 'Harvard University',
        type: 'INSTITUTION',
        createdAt: new Date().toISOString(),
        isLocked: true,
        data: {
            name: 'Harvard University',
            addressLine1: 'Massachusetts Hall',
            addressLine2: 'Cambridge, MA 02138',
            registrarName: 'Michael P. Smith',
            registrarTitle: 'University Registrar',
        }
    },
    {
        id: 'default-ucla',
        name: 'UCLA',
        type: 'INSTITUTION',
        createdAt: new Date().toISOString(),
        isLocked: true,
        data: {
            name: 'University of California, Los Angeles',
            addressLine1: '405 Hilgard Avenue',
            addressLine2: 'Los Angeles, CA 90095',
            registrarName: 'Sarah Jenkins',
            registrarTitle: 'Associate Dean of Records',
        }
    },
    {
        id: 'preset-standard',
        name: 'Standard Verification',
        type: 'PACKET_PRESET',
        createdAt: new Date().toISOString(),
        isLocked: true,
        data: {
            includedDocuments: ['transcript', 'enrollment_letter'],
            pdfSettings: {
                watermarkIntensity: 'MEDIUM',
                showPacketId: true,
                includePageNumbers: true,
                includeSectionDividers: true,
                includeTableOfContents: true
            }
        }
    }
];

export function useTemplates() {
  const [templates, setTemplates] = useState<Template[]>([]);
  const [isLoaded, setIsLoaded] = useState(false);

  // Load from Storage
  useEffect(() => {
    try {
      const stored = localStorage.getItem(STORAGE_KEY_TEMPLATES);
      let loaded: Template[] = stored ? JSON.parse(stored) : [];
      
      // Ensure defaults exist if list is empty or generic init
      if (loaded.length === 0) {
          loaded = DEFAULT_TEMPLATES;
          localStorage.setItem(STORAGE_KEY_TEMPLATES, JSON.stringify(loaded));
      } else {
        // Merge defaults in case they are missing (optional, but good for demo)
        const existingIds = new Set(loaded.map(t => t.id));
        const missingDefaults = DEFAULT_TEMPLATES.filter(d => !existingIds.has(d.id));
        if (missingDefaults.length > 0) {
            loaded = [...missingDefaults, ...loaded];
            localStorage.setItem(STORAGE_KEY_TEMPLATES, JSON.stringify(loaded));
        }
      }

      setTemplates(loaded);
      setIsLoaded(true);
    } catch (e) {
      console.error("Failed to load templates", e);
      setTemplates(DEFAULT_TEMPLATES);
      setIsLoaded(true);
    }
  }, []);

  // Save to Storage
  useEffect(() => {
    if (isLoaded) {
      localStorage.setItem(STORAGE_KEY_TEMPLATES, JSON.stringify(templates));
    }
  }, [templates, isLoaded]);

  const addTemplate = (template: Template) => {
    setTemplates(prev => [...prev, template]);
  };

  const deleteTemplate = (id: string) => {
    setTemplates(prev => prev.filter(t => t.id !== id));
  };

  const updatedTemplate = (id: string, updates: Partial<Template>) => {
      setTemplates(prev => prev.map(t => t.id === id ? { ...t, ...updates } as Template : t));
  };

  const getTemplatesByType = (type: TemplateType) => {
      return templates.filter(t => t.type === type);
  };

  return {
    templates,
    addTemplate,
    deleteTemplate,
    updatedTemplate,
    getTemplatesByType,
    isLoaded
  };
}
