"use client";

import { useState, useEffect } from 'react';
import { ServiceRecord, STORAGE_KEY_RECORDS, RecordType } from '@/types/records';
import { PacketRecord as LegacyPacketRecord } from '@/types/student-verification';

const ENABLE_DB_SYNC = process.env.NEXT_PUBLIC_ENABLE_DB_PERSISTENCE === 'true';

// Helper to migrate legacy student packets
const migrateLegacyRecords = (legacy: LegacyPacketRecord[]): ServiceRecord[] => {
  return legacy.map(rec => ({
    id: rec.id,
    systemRef: rec.systemRef,
    generatedAt: rec.generatedAt,
    type: 'STUDENT',
    status: 'GENERATED',
    details: {
      name: rec.studentName,
      institution: rec.institution,
      term: rec.term
    },
    data: rec.dataSnapshot,
    pdfSettings: rec.pdfSettingsSnapshot
  }));
};

export function useServiceRecords() {
  const [records, setRecords] = useState<ServiceRecord[]>([]);
  const [isLoaded, setIsLoaded] = useState(false);

  // Helper to safely save to localStorage with quota handling (hoisted)
  const saveToStorage = (recordsToSave: ServiceRecord[], updateState: boolean) => {
    try {
      localStorage.setItem(STORAGE_KEY_RECORDS, JSON.stringify(recordsToSave));
      if (updateState) setRecords(recordsToSave);
      return recordsToSave;
    } catch (e: any) {
      if (e.name === 'QuotaExceededError' || e.code === 22) {
        console.warn("Storage quota exceeded. Attempting to trim old records...");
        
        const saveAndSet = (trimmed: ServiceRecord[]) => {
            try {
                localStorage.setItem(STORAGE_KEY_RECORDS, JSON.stringify(trimmed));
                if (updateState) setRecords(trimmed);
                return trimmed;
            } catch (ignore) {
                return null;
            }
        };

        // Strategy 1: Keep only last 10
        if (recordsToSave.length > 10) {
            const result = saveAndSet(recordsToSave.slice(0, 10));
            if (result) return result;
        }
        
        // Strategy 2: Remove oldest one by one
        let retryQueue = [...recordsToSave];
        while (retryQueue.length > 0) {
            retryQueue.pop(); 
            const result = saveAndSet(retryQueue);
            if (result) return result;
        }
        
        console.error("Critical Storage Error: Cannot save even a single record.");
        return null;
      } else {
        console.error("Unknown Storage Error:", e);
        return null;
      }
    }
  };

  // Sync to Server Helper
  const syncToServer = async (record: ServiceRecord) => {
    if (!ENABLE_DB_SYNC) return;
    try {
        await fetch('/api/records', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ record }),
        });
    } catch (err) {
        console.error("Failed to sync record to server:", err);
    }
  };
  
  const deleteFromServer = async (id: string) => {
      if (!ENABLE_DB_SYNC) return;
      try {
          await fetch(`/api/records?id=${id}`, { method: 'DELETE' });
      } catch (err) {
          console.error("Failed to delete record from server:", err);
      }
  }

  // Load & Migrate on Mount
  useEffect(() => {
    const loadRecords = async () => {
        try {
          // 1. Load Main Storage (Local)
          const stored = localStorage.getItem(STORAGE_KEY_RECORDS);
          let loadedRecords: ServiceRecord[] = stored ? JSON.parse(stored) : [];
    
          // 2. Check & Migrate Legacy Student History
          const legacyStored = localStorage.getItem('genekei_student_packet_history');
          if (legacyStored) {
            const legacy: LegacyPacketRecord[] = JSON.parse(legacyStored);
            const migrated = migrateLegacyRecords(legacy);
            
            const existingIds = new Set(loadedRecords.map(r => r.id));
            const newToAd = migrated.filter(m => !existingIds.has(m.id));
            
            if (newToAd.length > 0) {
              loadedRecords = [...newToAd, ...loadedRecords];
              // Save migration to local
              const savedVersion = saveToStorage(loadedRecords, false);
              if (savedVersion) loadedRecords = savedVersion;
              
              // Also sync migrated to server if enabled? 
              // Doing simple sync of all on load is expensive.
              // Just let new additions sync for now.
            }
          }
           
          // 3. Load from DB if enabled
          if (ENABLE_DB_SYNC) {
              try {
                  const res = await fetch('/api/records');
                  if (res.ok) {
                      const data = await res.json();
                      if (data.ok && Array.isArray(data.records)) {
                          // Merge Strategy: Server wins? Or just Union?
                          // For now, simpler: Use Server list if available, as it's "Cloud"
                          // But we want to keep offline working.
                          // Let's Merge: Server records + Local records that aren't in Server
                          const serverRecords: ServiceRecord[] = data.records;
                          const serverIds = new Set(serverRecords.map(r => r.id));
                          
                          // Add local records that aren't on server (yet)
                          const localOnly = loadedRecords.filter(r => !serverIds.has(r.id));
                          
                          loadedRecords = [...serverRecords, ...localOnly];
                      }
                  }
              } catch (serverErr) {
                  console.warn("Could not fetch from server:", serverErr);
              }
          }
    
          // Sort by newest first
          loadedRecords.sort((a, b) => new Date(b.generatedAt).getTime() - new Date(a.generatedAt).getTime());
    
          setRecords(loadedRecords);
          setIsLoaded(true);
    
        } catch (e) {
          console.error("Failed to load service records:", e);
          setIsLoaded(true);
        }
    };
    
    loadRecords();
  }, []);

  // Save Effect for Updates (Local Only)
  useEffect(() => {
    if (isLoaded) {
      // Enforce hard limit of 20 before trying to save locally
      const recordsToSave = records.slice(0, 20);
      saveToStorage(recordsToSave, false); 
    }
  }, [records, isLoaded]);

  const addRecord = (record: ServiceRecord) => {
    setRecords(prev => {
        const updated = [record, ...prev];
        return updated;
    });
    // Async Server Sync
    syncToServer(record);
  };

  const updateRecord = (id: string, updates: Partial<ServiceRecord>) => {
    // We need the full record to sync to server (as we do full upsert)
    // So we find it in the current state
    const current = records.find(r => r.id === id);
    if (!current) return;
    
    const updatedRecord = { ...current, ...updates } as ServiceRecord;
    
    setRecords(prev => prev.map(rec => rec.id === id ? updatedRecord : rec));
    
    syncToServer(updatedRecord);
  };

  const deleteRecord = (id: string) => {
    setRecords(prev => prev.filter(rec => rec.id !== id));
    deleteFromServer(id);
  };

  const getRecord = (id: string) => records.find(r => r.id === id);

  // Search / Filter
  const searchRecords = (query: string, typeFilter?: RecordType) => {
    const lowerQuery = query.toLowerCase();
    
    return records.filter(rec => {
      // Type Filter
      if (typeFilter && rec.type !== typeFilter) return false;

      // Text Search
      if (!query) return true;

      const matchesRef = rec.systemRef.toLowerCase().includes(lowerQuery);
      
      let matchesDetails = false;
      if (rec.type === 'STUDENT') {
        matchesDetails = 
          rec.details.name.toLowerCase().includes(lowerQuery) || 
          rec.details.institution.toLowerCase().includes(lowerQuery) ||
          (rec.details.term || '').toLowerCase().includes(lowerQuery);
      } else if (rec.type === 'MILITARY') {
        matchesDetails = 
          rec.details.name.toLowerCase().includes(lowerQuery) || 
          rec.details.branch.toLowerCase().includes(lowerQuery);
      }

      return matchesRef || matchesDetails;
    });
  };

  return {
    records,
    isLoaded,
    addRecord,
    updateRecord,
    deleteRecord,
    getRecord,
    searchRecords
  };
}
