import React from 'react';
import { VerificationData } from '@/lib/generator';
import { formatDate } from '@/lib/dateHelpers';

// Hardcoded hex values for formal document style
const colors = {
    white: '#ffffff',
    black: '#000000',
    grayDark: '#333333',
    grayMed: '#666666',
    grayLight: '#e5e7eb', // For very subtle borders if needed
    border: '#000000',    // Strict black borders
};

// DoD Seal Placeholder (Simple circle with text for prototype)
const Seal = ({ logoUrl }: { logoUrl?: string }) => (
  logoUrl ? (
      <img src={logoUrl} alt="Official Seal" style={{ width: '80px', height: '80px', objectFit: 'contain' }} />
  ) : (
    <div style={{ width: '80px', height: '80px', borderRadius: '50%', border: '4px solid #000', display: 'flex', alignItems: 'center', justifyContent: 'center', margin: '0 auto' }}>
        <div style={{ textAlign: 'center', fontSize: '10px', fontWeight: 'bold', lineHeight: '1' }}>
            OFFICIAL<br/>SEAL
        </div>
    </div>
  )
);

const Barcode = () => (
    <div style={{ display: 'flex', alignItems: 'center', flexDirection: 'column' }}>
        <div style={{ display: 'flex', height: '30px', alignItems: 'flex-end' }}>
            {Array.from({ length: 40 }).map((_, i) => (
                <div key={i} style={{ 
                    width: (i * 13) % 3 === 0 ? '4px' : '2px', 
                    height: '100%', 
                    backgroundColor: 'black', 
                    marginRight: (i * 7) % 2 === 0 ? '1px' : '2px',
                    display: 'inline-block'
                }} />
            ))}
        </div>
        <div style={{ fontSize: '8px', fontFamily: 'Courier New, monospace', letterSpacing: '2px' }} suppressHydrationWarning>
            {Math.floor(Math.random() * 1000000000)}
        </div>
    </div>
);

const Box = ({ label, value, className = '', highlight = false }: { label: string, value?: string, className?: string, highlight?: boolean }) => (
    <div 
      className={className} 
      style={{ 
          borderRight: '1px solid black', 
          borderBottom: '1px solid black', 
          padding: '4px 8px',
          backgroundColor: highlight ? '#f3f4f6' : 'transparent', 
          minHeight: '48px',
          display: 'flex', 
          flexDirection: 'column',
          justifyContent: 'space-between'
      }}
    >
        <div style={{ fontSize: '9px', textTransform: 'uppercase', fontWeight: 'bold', color: colors.grayDark, lineHeight: '1.2' }}>{label}</div>
        <div style={{ fontSize: '12px', fontFamily: 'Times New Roman, serif', fontWeight: 'bold', color: colors.black, lineHeight: '1.2', marginTop: '2px' }}>
            {value || '—'}
        </div>
    </div>
);

export function VerifiedPdfTemplate({ data }: { readonly data: Partial<VerificationData> }) {

  const serviceDuration = getDurationString(data.startDate, data.dischargeDate);
  const transitionStatus = getTransitionStatus(data.status, data.dischargeDate);

  // Container with security background pattern
    const containerStyle: React.CSSProperties = {
        width: '210mm',
        minHeight: '297mm',
        padding: '20mm',
        margin: '0 auto',
        backgroundColor: 'white',
        color: 'black',
        fontFamily: 'Arial, sans-serif',
        boxSizing: 'border-box',
        position: 'relative',
        backgroundImage: 'repeating-linear-gradient(45deg, transparent, transparent 10px, rgba(0,0,0,0.03) 10px, rgba(0,0,0,0.03) 12px), repeating-linear-gradient(-45deg, transparent, transparent 10px, rgba(0,0,0,0.03) 10px, rgba(0,0,0,0.03) 12px)',
    };

    if (!data) return null;

  return (
    <div id="pdf-template" style={containerStyle}>
      
      {/* HEADER */}
      <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', marginBottom: '30px', borderBottom: '4px double black', paddingBottom: '20px' }}>
         <div style={{ flex: 1 }}><Seal logoUrl={data.logo} /></div>
         <div style={{ textAlign: 'center', flex: 2 }}>
            <h1 style={{ fontSize: '24px', textTransform: 'uppercase', fontWeight: '900', letterSpacing: '1px', marginBottom: '4px', fontFamily: 'Times New Roman, serif' }}>Department of Defense</h1>
            <h2 style={{ fontSize: '16px', textTransform: 'uppercase', fontWeight: 'bold', marginBottom: '0' }}>Certificate of Release or Discharge from Active Duty</h2>
            <p style={{ fontSize: '10px', marginTop: '4px', fontStyle: 'italic' }}>Verified Service Verification System Reference Copy</p>
         </div>
         <div style={{ flex: 1, display: 'flex', justifyContent: 'flex-end' }}>
             <Barcode />
         </div>
      </div>

      {/* FORM BODY */}
      <div style={{ border: '2px solid black', borderRight: '1px solid black', borderBottom: '1px solid black' }}> 
        {/* Row 1: Name and Personal Data */}
        <div style={{ display: 'flex' }}>
            {data.photo && (
                <div style={{ width: '100px', borderRight: '1px solid black', borderBottom: '1px solid black', overflow: 'hidden' }}>
                    <img src={data.photo} alt="Member" style={{ width: '100%', height: '100%', objectFit: 'cover', display: 'block' }} />
                </div>
            )}
            <Box label="1. Name (Last, First, Middle)" value={`${data.lastName?.toUpperCase() || ''}, ${data.firstName?.toUpperCase() || ''}`} className={data.photo ? "flex-[2]" : "flex-[3]"} />
            <Box label="2. Branch of Service" value={data.branch?.toUpperCase()} className="flex-[2]" />
            <Box label="3. Status" value={data.status?.toUpperCase()} className="flex-[2]" />
        </div>

        {/* Row 2: Dates */}
        <div style={{ display: 'flex' }}>
            <Box label="4a. Date of Birth" value={formatDate(data.dob)} className="flex-1" />
            <Box label="4b. Service Start Date" value={formatDate(data.startDate)} className="flex-1" />
            <Box label="4c. Separation Date" value={formatDate(data.dischargeDate)} className="flex-1" />
        </div>

        {/* Row 3: Contact and Recruitment */}
        <div style={{ display: 'flex' }}>
            <Box label="5. Email Address" value={data.email} className="flex-[2]" />
            <Box label="6. Place of Entry (Recruitment Center)" value={data.recruitmentCenter} className="flex-[2]" />
        </div>

        {/* Section Header */}
        <div style={{ backgroundColor: '#e5e7eb', borderBottom: '1px solid black', borderRight: '1px solid black', padding: '4px 8px', fontSize: '11px', fontWeight: 'bold', textTransform: 'uppercase', textAlign: 'center', letterSpacing: '1px' }}>
            Record of Service
        </div>

        {/* Row 4: Service Calculation */}
        <div style={{ display: 'flex' }}>
            <div className="flex-1" style={{ borderRight: '1px solid black', borderBottom: '1px solid black', padding: '0' }}>
               <div style={{ padding: '4px 8px', borderBottom: '1px solid black', display: 'flex', justifyContent: 'space-between', fontSize: '10px' }}>
                   <span>a. TOTAL NET SERVICE</span>
                   <span style={{ fontFamily: 'Times New Roman, serif', fontWeight: 'bold' }}>{serviceDuration.toUpperCase()}</span>
               </div>
               <div style={{ padding: '4px 8px', borderBottom: '1px solid black', display: 'flex', justifyContent: 'space-between', fontSize: '10px' }}>
                   <span>b. FOREIGN SERVICE</span>
                   <span style={{ fontFamily: 'Times New Roman, serif', fontWeight: 'bold' }}>0 YEARS, 0 MONTHS</span>
               </div>
               <div style={{ padding: '4px 8px', display: 'flex', justifyContent: 'space-between', fontSize: '10px' }}>
                   <span>c. SEA SERVICE</span>
                   <span style={{ fontFamily: 'Times New Roman, serif', fontWeight: 'bold' }}>0 YEARS, 0 MONTHS</span>
               </div>
            </div>
            
            {/* Status Box */}
            <div className="flex-1" style={{ borderRight: '1px solid black', borderBottom: '1px solid black', padding: '20px', display: 'flex', alignItems: 'center', justifyContent: 'center', flexDirection: 'column' }}>
                <div style={{ fontSize: '10px', textTransform: 'uppercase', fontWeight: 'bold', marginBottom: '8px' }}>Transition Status Verification</div>
                {transitionStatus ? (
                     <div style={{ 
                         border: '2px solid black', 
                         padding: '8px 16px', 
                         textAlign: 'center',
                         backgroundColor: transitionStatus.backgroundColor,
                     }}>
                         <div style={{ fontWeight: 'bold', fontSize: '14px', color: 'black' }}>{transitionStatus.label}</div>
                     </div>
                ) : (
                    <div style={{ fontStyle: 'italic', fontSize: '12px', color: '#666' }}>No Special Status Indicated</div>
                )}
            </div>
        </div>

        {/* Row 5: REMARKS */}
        <div style={{ borderBottom: '1px solid black', height: '140px', padding: '4px 8px' }}>
            <div style={{ fontSize: '9px', textTransform: 'uppercase', fontWeight: 'bold', color: colors.grayDark, marginBottom: '6px' }}>18. REMARKS</div>
            <div style={{ fontSize: '11px', fontFamily: 'Times New Roman, serif', lineHeight: '1.4', textTransform: 'uppercase' }}>
                IMMEDIATE REENLISTMENT THIS DATE.<br/>
                MEMBER CONTRIBUTED TO POST-VIETNAM ERA VETERANS' EDUCATIONAL ASSISTANCE PROGRAM.<br/>
                TIME LOST: NONE.<br/>
                DENTAL EXAM COMPLETED AND ALL TREATMENT SATISFACTORY.<br/>
                <br/>
                // NOTHING FOLLOWS //
            </div>
        </div>

        {/* BOTTOM SECTION: BLOCKS 23-30 */}
        <div style={{ backgroundColor: '#e5e7eb', borderBottom: '1px solid black', borderRight: '1px solid black', padding: '4px 8px', fontSize: '11px', fontWeight: 'bold', textTransform: 'uppercase', textAlign: 'center', letterSpacing: '1px' }}>
            Special Additional Information
        </div>
        
        <div style={{ display: 'flex' }}>
            <Box label="23. TYPE OF SEPARATION" value="DISCHARGE" className="flex-[2]" />
            <Box label="24. CHARACTER OF SERVICE" value={data.status === 'Active Duty' ? "HONORABLE" : "HONORABLE"} className="flex-[2]" />
            <Box label="25. SEPARATION AUTHORITY" value="AR 635-200, CHAP 4" className="flex-[2]" />
        </div>
        <div style={{ display: 'flex' }}>
            <Box label="26. SEPARATION CODE" value="JBK" className="flex-1" />
            <Box label="27. REENTRY CODE" value="RE-1" className="flex-1" />
            <Box label="28. NARRATIVE REASON FOR SEPARATION" value="COMPLETION OF REQUIRED ACTIVE SERVICE" className="flex-[2]" />
        </div>
        <div style={{ display: 'flex' }}>
             <Box label="29. DATES OF TIME LOST DURING THIS PERIOD" value="NONE" className="flex-[2]" />
             <Box label="30. MEMBER REQUESTS COPY 4" value="YES" className="flex-1" />
        </div>

      </div>

      {/* FOOTER */}
      <div style={{ marginTop: '20px', display: 'flex', justifyContent: 'space-between', alignItems: 'end', fontSize: '10px', fontFamily: 'Times New Roman, serif' }}>
          <div style={{ width: '40%', position: 'relative', marginBottom: '20px' }}>
              <div style={{ 
                  position: 'absolute', 
                  bottom: '15px', 
                  left: '20px',
                  fontFamily: '"Brush Script MT", "Segoe Script", cursive', 
                  fontSize: '24px', 
                  lineHeight: '1',
                  whiteSpace: 'nowrap'
              }}>
                  {data.firstName} {data.lastName}
              </div>
              <div style={{ borderTop: '1px solid black', width: '100%', paddingTop: '4px' }}>
                  MEMBER SIGNATURE
              </div>
          </div>
          <div style={{ width: '40%', textAlign: 'right', position: 'relative', marginBottom: '20px' }}>
              <div style={{ 
                  position: 'absolute', 
                  bottom: '15px', 
                  right: '20px',
                  fontFamily: '"Brush Script MT", "Segoe Script", cursive', 
                  fontSize: '24px', 
                  lineHeight: '1',
                  whiteSpace: 'nowrap'
              }}>
                  {data.authorizedOfficialName || 'GEN. A. Official'}
              </div>
              <div style={{ borderTop: '1px solid black', width: '100%', paddingTop: '4px' }}>
                  OFFICIAL AUTHORIZED SIGNATURE
              </div>
          </div>
      </div>
      
      {/* BOTTOM METADATA */}


      {/* PRIVACY ACT STATEMENT */}
      <div style={{ marginTop: '30px', borderTop: '1px solid #ccc', paddingTop: '4px', fontSize: '7px', textAlign: 'justify', color: '#666' }}>
          AUTHORITY: 10 U.S.C. 1168. PRINCIPAL PURPOSE: To record transfer or discharge. ROUTINE USES: To the Dept of Veterans Affairs. DISCLOSURE: Voluntary; however, failure to provide info may result in non-issuance.
      </div>
    
    </div>
  );
}


function getDurationString(start?: Date | null, end?: Date | null): string {
    if (!start || !end) return '—';
    
    // Simple diff
    let months = (end.getFullYear() - start.getFullYear()) * 12;
    months -= start.getMonth();
    months += end.getMonth();
    
    if (months <= 0) return '0 Months';

    const years = Math.floor(months / 12);
    const remainingMonths = months % 12;

    const parts = [];
    if (years > 0) parts.push(`${years} Year${years !== 1 ? 's' : ''}`);
    if (remainingMonths > 0) parts.push(`${remainingMonths} Month${remainingMonths !== 1 ? 's' : ''}`);
    
    return parts.join(', ');
}

interface StatusBadge {
    label: string;
    backgroundColor: string;
    textColor: string;
    borderColor: string;
}

function getTransitionStatus(status?: string, dischargeDate?: Date | null): StatusBadge | null {
    if (!status || !dischargeDate) return null;

    const now = new Date();
    const diffTime = dischargeDate.getTime() - now.getTime();
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)); 
    
    // Logic:
    // 1. Veteran separated < 12 months ago
    // 2. Active Duty separating within < 12 months (future)

    const isVeteran = status.includes('Veteran') || status.includes('Discharged') || status.includes('Retired');
    const isActive = status.includes('Active');

    // 12 months in days approx 365
    if (isVeteran) {
        // Discharge date is in the past. 
        // If diffDays is negative (past), check if it's within -365 days
        if (diffDays < 0 && diffDays > -365) {
             return { 
                 label: 'Recently Separated (< 12 Months)', 
                 backgroundColor: '#f0fdf4', // green-50
                 textColor: '#15803d',       // green-700
                 borderColor: '#bbf7d0'      // green-200
             };
        }
    }

    if (isActive) {
        // Discharge date is in future
        if (diffDays >= 0 && diffDays < 365) {
             return {
                 label: 'Transitioning (Separation < 12 Months)',
                 backgroundColor: '#fefce8', // yellow-50
                 textColor: '#a16207',       // yellow-700
                 borderColor: '#fde68a'      // yellow-200
             };
        }
    }

    return null;
}

