"use client";

import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/Card';
import { Label } from '@/components/ui/Label';
import { Input } from '@/components/ui/Input';
import { Select } from '@/components/ui/Select';
import { Button } from '@/components/ui/Button';
import { DateSelect } from '@/components/DateSelect';
import { STATUS_OPTIONS, BRANCH_OPTIONS } from '@/lib/constants';
import { validateVerificationData } from '@/lib/validator';
import { generateProfile, VerificationData } from '@/lib/generator';
import { RefreshCw, Loader2 } from 'lucide-react';
import { pdf } from '@react-pdf/renderer';
import { VectorPdfDocument } from './VectorPdfDocument';

export function VerificationForm() {
  const [formData, setFormData] = useState<Partial<VerificationData>>({});
  const [isGenerating, setIsGenerating] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [successMessage, setSuccessMessage] = useState<string | null>(null);

  const handleAutofill = () => {
    setIsGenerating(true);
    setSuccessMessage(null);
    // Simulate a small delay for "feel"
    setTimeout(() => {
      const data = generateProfile();
      setFormData(data);
      setErrors({});
      setIsGenerating(false);
    }, 400);
  };

  const handleClear = () => {
    setFormData({});
    setErrors({});
    setSuccessMessage(null);
  };

  const handleChange = (field: keyof VerificationData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    // Clear error for this field
    if (errors[field]) {
      setErrors(prev => {
        const newErrors = { ...prev };
        delete newErrors[field];
        return newErrors;
      });
    }
  };

  const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>, field: 'logo' | 'photo') => {
      const file = event.target.files?.[0];
      if (file) {
          const reader = new FileReader();
          reader.onloadend = () => {
              handleChange(field, reader.result as string);
          };
          reader.readAsDataURL(file);
      }
  };

  const handleDownload = () => {
    // Validate
    const newErrors = validateVerificationData(formData);

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors);
      return;
    }

    handleGeneratePdf();
  };

  const handleGeneratePdf = async () => {
    setIsGenerating(true);

    try {
        // Generate Blob using @react-pdf/renderer
        const blob = await pdf(<VectorPdfDocument data={formData} />).toBlob();
        
        // Create download link
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = `Service_Verification_${formData.lastName || 'Sample'}.pdf`;
        document.body.appendChild(link);
        link.click();
        link.remove();
        URL.revokeObjectURL(url);

        setIsGenerating(false);
        setSuccessMessage('PDF Downloaded Successfully!');
    } catch (err: any) {
        console.error('PDF Generation Error:', err);
        setIsGenerating(false);
        // Show more detailed error details
        alert(`Error generating PDF: ${err.message || err.toString()}`);
    }
  };

  return (
    <div className="mx-auto grid max-w-2xl gap-6 py-8 relative">
      <div className="flex items-center justify-between">
        <h1 className="text-xl font-semibold text-gray-900">DATA GENERATOR</h1>
        <Button onClick={handleAutofill} disabled={isGenerating}>
           <RefreshCw className={`mr-2 h-4 w-4 ${isGenerating ? 'animate-spin' : ''}`} />
           Autofill
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="text-2xl">Military Verification</CardTitle>
          <p className="text-right text-xs text-gray-500">* Required information</p>
        </CardHeader>
        <CardContent className="space-y-6">
          <div className="space-y-2">
            <Label required>Status</Label>
            <Select 
              value={formData.status || ''} 
              onChange={(e) => handleChange('status', e.target.value)}
              error={!!errors.status}
              aria-label="Status"
              disabled={isGenerating}
            >
              <option value="" disabled>Select status</option>
              {STATUS_OPTIONS.map(opt => <option key={opt} value={opt}>{opt}</option>)}
            </Select>
            {errors.status && <p className="text-xs text-red-500">{errors.status}</p>}
          </div>

          <div className="space-y-2">
            <Label required>Branch of service</Label>
            <Select 
              value={formData.branch || ''} 
              onChange={(e) => handleChange('branch', e.target.value)}
              error={!!errors.branch}
              aria-label="Branch"
              disabled={isGenerating}
            >
              <option value="" disabled>Select branch</option>
              {BRANCH_OPTIONS.map(opt => <option key={opt} value={opt}>{opt}</option>)}
            </Select>
            {errors.branch && <p className="text-xs text-red-500">{errors.branch}</p>}
          </div>

          <div className="grid gap-4 sm:grid-cols-2">
            <div className="space-y-2">
              <Label required>First name</Label>
              <Input 
                 value={formData.firstName || ''} 
                 onChange={(e) => handleChange('firstName', e.target.value)}
                 error={!!errors.firstName}
                 disabled={isGenerating}
              />
              {errors.firstName && <p className="text-xs text-red-500">{errors.firstName}</p>}
            </div>
            <div className="space-y-2">
              <Label required>Last name</Label>
              <Input 
                 value={formData.lastName || ''} 
                 onChange={(e) => handleChange('lastName', e.target.value)}
                 error={!!errors.lastName}
                 disabled={isGenerating}
              />
              {errors.lastName && <p className="text-xs text-red-500">{errors.lastName}</p>}
            </div>
          </div>

          <div className="space-y-2">
            <Label required>Date of birth</Label>
            <p className="text-xs text-gray-500">Used for verification purposes only</p>
            <DateSelect 
              value={formData.dob} 
              onChange={(date) => handleChange('dob', date)}
              startYear={1920}
              endYear={new Date().getFullYear() - 15}
              disabled={isGenerating}
            />
            {errors.dob && <p className="text-xs text-red-500">{errors.dob}</p>}
          </div>

          <div className="space-y-2">
             <Label required>Start date</Label>
             <p className="text-xs text-gray-500">Used for verification purposes only</p>
             <DateSelect 
              value={formData.startDate} 
              onChange={(date) => handleChange('startDate', date)}
              startYear={1950}
              disabled={isGenerating}
            />
             {errors.startDate && <p className="text-xs text-red-500">{errors.startDate}</p>}
          </div>

          <div className="space-y-2">
            <Label required>Recruitment Center</Label>
            <p className="text-xs text-gray-500">Where you enlisted (e.g., New York MEPS)</p>
            <Input 
                value={formData.recruitmentCenter || ''} 
                onChange={(e) => handleChange('recruitmentCenter', e.target.value)}
                error={!!errors.recruitmentCenter}
                disabled={isGenerating}
            />
            {errors.recruitmentCenter && <p className="text-xs text-red-500">{errors.recruitmentCenter}</p>}
          </div>
          
           <div className="space-y-2">
             <Label required>Discharge date</Label>
             <p className="text-xs text-gray-500">Used for verification purposes only</p>
             <DateSelect 
              value={formData.dischargeDate} 
              onChange={(date) => handleChange('dischargeDate', date)}
              startYear={1950}
              endYear={new Date().getFullYear() + 20}
              disabled={isGenerating}
            />
            {errors.dischargeDate && <p className="text-xs text-red-500">{errors.dischargeDate}</p>}
          </div>

          <div className="space-y-2">
            <Label required>Email address</Label>
            <p className="text-xs text-gray-500">Personal email address is recommended</p>
            <Input 
               value={formData.email || ''} 
               onChange={(e) => handleChange('email', e.target.value)}
               error={!!errors.email}
               placeholder="name@example.com"
               disabled={isGenerating}
            />
             {errors.email && <p className="text-xs text-red-500">{errors.email}</p>}
          </div>

          <div className="pt-4 border-t border-gray-100">
              <h3 className="text-sm font-medium text-gray-900 mb-4">Customizations (Optional)</h3>
              <div className="grid gap-4 sm:grid-cols-2">
                 <div className="space-y-2">
                    <Label>Agency Logo</Label>
                    <p className="text-xs text-gray-500">Replaces the official seal (PNG/JPG)</p>
                    <Input 
                        type="file" 
                        accept="image/*"
                        onChange={(e) => handleImageUpload(e, 'logo')}
                        disabled={isGenerating}
                    />
                 </div>
                 <div className="space-y-2">
                    <Label>Member Photo</Label>
                    <p className="text-xs text-gray-500">Displayed in personal details (PNG/JPG)</p>
                    <Input 
                        type="file" 
                        accept="image/*"
                        onChange={(e) => handleImageUpload(e, 'photo')}
                        disabled={isGenerating}
                    />
                 </div>
              </div>
          </div>

        </CardContent>
      </Card>
      
      {successMessage && (
        <div className="w-full text-center p-2 bg-green-50 text-green-700 rounded-md border border-green-200 text-sm font-medium animate-in fade-in slide-in-from-top-2">
            {successMessage}
        </div>
      )}

      <div className="flex flex-col gap-3 sm:flex-row">
           <Button className="w-full flex-1" variant="primary" size="lg" onClick={handleDownload} disabled={isGenerating}>
              {isGenerating ? <span className="flex items-center"><Loader2 className="mr-2 h-4 w-4 animate-spin" /> Generating...</span> : 'Verify & Download PDF'}
           </Button>
            <Button className="w-full sm:w-auto" variant="secondary" size="lg" onClick={handleClear} disabled={isGenerating}>
              Clear
           </Button>
      </div>

    </div>
  );
}
