import React from 'react';
import { Page, Text, View, Document, StyleSheet, Image } from '@react-pdf/renderer';
import { VerificationData } from '@/lib/generator';
import { formatDate } from '@/lib/dateHelpers';

// Register fonts (standard PDF fonts are built-in, but we can register others if needed)
// Using standard fonts for now to ensure compatibility: Times-Roman, Helvetica, Courier

const styles = StyleSheet.create({
  page: {
    flexDirection: 'column',
    backgroundColor: '#FFFFFF',
    padding: 40, // Approx 20mm
    fontFamily: 'Helvetica',
    fontSize: 10,
    position: 'relative',
  },
  
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 20,
    borderBottomWidth: 4,
    borderBottomStyle: 'solid',
    borderBottomColor: 'black',
    paddingBottom: 10,
    height: 100,
  },
  sealContainer: {
    width: 80,
    height: 80,
    alignItems: 'center',
    justifyContent: 'center',
  },
  sealImage: {
    width: 80,
    height: 80,
    objectFit: 'contain',
  },
  sealPlaceholder: {
    width: 80,
    height: 80,
    borderRadius: 40,
    borderWidth: 4,
    borderColor: 'black',
    alignItems: 'center',
    justifyContent: 'center',
  },
  sealText: {
    textAlign: 'center',
    fontSize: 10,
    fontWeight: 'bold',
  },
  headerTextContainer: {
    flex: 1, // Take remaining space
    alignItems: 'center',
    paddingHorizontal: 5,
  },
  headerTitle: {
    fontSize: 16,
    textTransform: 'uppercase',
    fontFamily: 'Times-Bold',
    marginBottom: 4,
    letterSpacing: 1,
  },
  headerSubtitle: {
    fontSize: 9,
    textTransform: 'uppercase',
    fontFamily: 'Helvetica-Bold',
    marginBottom: 0,
    textAlign: 'center',
  },
  headerRef: {
    fontSize: 10,
    marginTop: 4,
    fontFamily: 'Helvetica-Oblique',
  },
  barcodeContainer: {
    width: 140, // Fixed width to prevent overlap
    alignItems: 'flex-end',
    justifyContent: 'flex-start',
  },
  barcode: {
    flexDirection: 'row',
    alignItems: 'flex-end',
    height: 30,
  },
  barcodeText: {
    fontSize: 8,
    fontFamily: 'Courier',
    letterSpacing: 2,
    marginTop: 2,
  },
  
  // Table/Grid styles
  section: {
    borderWidth: 2,
    borderColor: 'black',
    borderRightWidth: 1,
    borderBottomWidth: 1,
  },
  row: {
    flexDirection: 'row',
    borderBottomWidth: 0, // Handled by cells
  },
  cell: {
    borderRightWidth: 1,
    borderBottomWidth: 1,
    borderColor: 'black',
    padding: 4,
    flexDirection: 'column',
    justifyContent: 'space-between',
    minHeight: 40,
  },
  cellLabel: {
    fontSize: 8,
    textTransform: 'uppercase',
    color: '#333333',
    fontFamily: 'Helvetica-Bold',
  },
  cellValue: {
    fontSize: 11,
    fontFamily: 'Times-Bold',
    marginTop: 2,
  },
  
  // Section Headers
  sectionHeader: {
    backgroundColor: '#e5e7eb',
    borderBottomWidth: 1,
    borderRightWidth: 1,
    borderColor: 'black',
    padding: 4,
    textAlign: 'center',
  },
  sectionTitle: {
    fontSize: 10,
    fontFamily: 'Helvetica-Bold',
    textTransform: 'uppercase',
    letterSpacing: 1,
  },

  // Service Record
  serviceRecordRow: {
    flexDirection: 'row',
  },
  serviceCol: {
    flex: 1,
    borderRightWidth: 1,
    borderColor: 'black',
  },
  serviceItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    padding: 4,
    borderBottomWidth: 1,
    borderColor: 'black',
    fontSize: 10,
  },
  
  // Remarks
  remarksSection: {
    height: 140,
    padding: 4,
    borderBottomWidth: 1,
    borderRightWidth: 1, // Fixed border
    borderColor: 'black',
  },
  remarksText: {
    fontSize: 11,
    fontFamily: 'Times-Roman',
    lineHeight: 1.4,
    textTransform: 'uppercase',
  },

  // Footer
  footer: {
    marginTop: 20,
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-end',
  },
  signatureBlock: {
    width: '40%',
  },
  signatureText: {
    fontFamily: 'Times-Italic', // Generating cursive-like look
    fontSize: 18,
    marginBottom: 4,
    paddingLeft: 10,
  },
  signatureLine: {
    borderTopWidth: 1,
    borderColor: 'black',
    paddingTop: 4,
  },
  signatureLabel: {
    fontSize: 10,
    fontFamily: 'Times-Roman',
  },

  bottomMeta: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: 10,
    fontSize: 8,
    alignItems: 'flex-end',
  },
  
  privacyAct: {
    marginTop: 20,
    borderTopWidth: 1,
    borderColor: '#ccc',
    paddingTop: 4,
    fontSize: 7,
    color: '#666',
    textAlign: 'justify',
  },

  // Helpers
  flex1: { flex: 1 },
  flex2: { flex: 2 },
  flex3: { flex: 3 },
});

// Helper for duration
function getDurationString(start?: Date | null, end?: Date | null): string {
    if (!start || !end) return '—';
    let months = (new Date(end).getFullYear() - new Date(start).getFullYear()) * 12;
    months -= new Date(start).getMonth();
    months += new Date(end).getMonth();
    
    if (months <= 0) return '0 Months';

    const years = Math.floor(months / 12);
    const remainingMonths = months % 12;

    const parts = [];
    if (years > 0) parts.push(`${years} Year${years !== 1 ? 's' : ''}`);
    if (remainingMonths > 0) parts.push(`${remainingMonths} Month${remainingMonths !== 1 ? 's' : ''}`);
    
    return parts.join(', ');
}

interface VectorPdfProps {
  data: Partial<VerificationData>;
}

export const VectorPdfDocument: React.FC<VectorPdfProps> = ({ data }) => {
  const serviceDuration = getDurationString(data.startDate, data.dischargeDate);
  // Status check for badge color simulation (backgrounds supported in Views)
  const isVeteran = data.status?.includes('Veteran');
  
  function getRandomId(name?: string) {
      if (!name) return 12534567;
      return Math.floor(100000000 + name.length * 827261);
  }
  
  // Deterministic random ID
  const randomId = getRandomId(data.lastName);

  const barcodeBars = Array.from({ length: 30 }).map((_, i) => ({
      width: (i * 13) % 3 === 0 ? 4 : 2,
      marginRight: (i * 7) % 2 === 0 ? 1 : 2,
  }));

  return (
    <Document>
      <Page size="LETTER" style={styles.page}>
        
        {/* Header */}
        <View style={styles.header}>
            <View style={{ width: 90 }}>
                {data.logo ? (
                    <Image src={data.logo} style={styles.sealImage} />
                ) : (
                    <View style={styles.sealPlaceholder}>
                        <Text style={styles.sealText}>OFFICIAL{'\n'}SEAL</Text>
                    </View>
                )}
            </View>
            <View style={styles.headerTextContainer}>
                <Text style={styles.headerTitle}>Department of Defense</Text>
                <Text style={styles.headerSubtitle}>Certificate of Release or Discharge</Text>
                <Text style={styles.headerSubtitle}>From Active Duty</Text>
                <Text style={styles.headerRef}>Verified Service Verification System Reference Copy</Text>
            </View>
            <View style={styles.barcodeContainer}>
                <View style={styles.barcode}>
                    {barcodeBars.map((bar, i) => (
                        <View key={`b${i}`} style={{ width: bar.width, height: '100%', backgroundColor: 'black', marginRight: bar.marginRight }} />
                    ))}
                </View>
                <Text style={styles.barcodeText}>{randomId}</Text>
            </View>
        </View>

        {/* Body Border info */}
        <View style={styles.section}>
            
            {/* Row 1 */}
            <View style={styles.row}>
                {data.photo && (
                    <View style={{ width: 100, borderRightWidth: 1, borderBottomWidth: 1, borderColor: 'black' }}>
                         <Image src={data.photo} style={{ width: '100%', height: 100, objectFit: 'cover' }} />
                    </View>
                )}
                <View style={[styles.cell, data.photo ? styles.flex2 : styles.flex3 ]}>
                    <Text style={styles.cellLabel}>1. Name (Last, First, Middle)</Text>
                    <Text style={styles.cellValue}>{data.lastName?.toUpperCase()}, {data.firstName?.toUpperCase()}</Text>
                </View>
                <View style={[styles.cell, styles.flex2]}>
                    <Text style={styles.cellLabel}>2. Branch of Service</Text>
                    <Text style={styles.cellValue}>{data.branch?.toUpperCase()}</Text>
                </View>
                <View style={[styles.cell, styles.flex2]}>
                    <Text style={styles.cellLabel}>3. Status</Text>
                    <Text style={styles.cellValue}>{data.status?.toUpperCase()}</Text>
                </View>
            </View>

            {/* Row 2 */}
            <View style={styles.row}>
                <View style={[styles.cell, styles.flex1]}>
                    <Text style={styles.cellLabel}>4a. Date of Birth</Text>
                    <Text style={styles.cellValue}>{formatDate(data.dob)}</Text>
                </View>
                <View style={[styles.cell, styles.flex1]}>
                    <Text style={styles.cellLabel}>4b. Service Start Date</Text>
                    <Text style={styles.cellValue}>{formatDate(data.startDate)}</Text>
                </View>
                <View style={[styles.cell, styles.flex1]}>
                    <Text style={styles.cellLabel}>4c. Separation Date</Text>
                    <Text style={styles.cellValue}>{formatDate(data.dischargeDate)}</Text>
                </View>
            </View>

            {/* Row 3 */}
            <View style={styles.row}>
                <View style={[styles.cell, styles.flex2]}>
                     <Text style={styles.cellLabel}>5. Email Address</Text>
                     <Text style={styles.cellValue}>{data.email}</Text>
                </View>
                <View style={[styles.cell, styles.flex2]}>
                     <Text style={styles.cellLabel}>6. Place of Entry (Recruitment Center)</Text>
                     <Text style={styles.cellValue}>{data.recruitmentCenter}</Text>
                </View>
            </View>

            {/* Section Header */}
            <View style={styles.sectionHeader}>
                <Text style={styles.sectionTitle}>Record of Service</Text>
            </View>

             {/* Row 4 */}
            <View style={styles.serviceRecordRow}>
                <View style={[styles.serviceCol, styles.flex2]}>
                    <View style={styles.serviceItem}>
                        <Text>a. TOTAL NET SERVICE</Text>
                        <Text style={{ fontFamily: 'Times-Bold' }}>{serviceDuration.toUpperCase()}</Text>
                    </View>
                    <View style={styles.serviceItem}>
                        <Text>b. FOREIGN SERVICE</Text>
                        <Text style={{ fontFamily: 'Times-Bold' }}>0 YEARS, 0 MONTHS</Text>
                    </View>
                     <View style={[styles.serviceItem, { borderBottomWidth: 0 }]}>
                        <Text>c. SEA SERVICE</Text>
                        <Text style={{ fontFamily: 'Times-Bold' }}>0 YEARS, 0 MONTHS</Text>
                    </View>
                </View>
                {/* Status verification box */}
                <View style={[styles.serviceCol, styles.flex1, { alignItems: 'center', justifyContent: 'center', padding: 10 }]}>
                     <Text style={[styles.cellLabel, { marginBottom: 5 }]}>Transition Status</Text>
                     <View style={{ 
                         backgroundColor: isVeteran ? '#f0fdf4' : '#fefce8', 
                         padding: '4 8', 
                         borderWidth: 1,
                     }}>
                         <Text style={{ fontSize: 10, fontFamily: 'Helvetica-Bold' }}>
                             {isVeteran ? 'Recently Separated' : 'Transitioning'}
                         </Text>
                     </View>
                </View>
            </View>

            {/* Row 5 Remarks */}
            <View style={styles.remarksSection}>
                <Text style={styles.cellLabel}>18. REMARKS</Text>
                <Text style={[styles.remarksText, { marginTop: 4 }]}>
                    IMMEDIATE REENLISTMENT THIS DATE.{'\n'}
                    MEMBER CONTRIBUTED TO POST-VIETNAM ERA VETERANS' EDUCATIONAL ASSISTANCE PROGRAM.{'\n'}
                    TIME LOST: NONE.{'\n'}
                    DENTAL EXAM COMPLETED AND ALL TREATMENT SATISFACTORY.{'\n'}
                    {'\n'}
                    {'// NOTHING FOLLOWS //'}
                </Text>
            </View>

            {/* Bottom Section Header */}
            <View style={styles.sectionHeader}>
                <Text style={styles.sectionTitle}>Special Additional Information</Text>
            </View>

             {/* Bottom Rows */}
            <View style={styles.row}>
                <View style={[styles.cell, styles.flex2]}>
                     <Text style={styles.cellLabel}>23. TYPE OF SEPARATION</Text>
                     <Text style={styles.cellValue}>DISCHARGE</Text>
                </View>
                 <View style={[styles.cell, styles.flex2]}>
                     <Text style={styles.cellLabel}>24. CHARACTER OF SERVICE</Text>
                     <Text style={styles.cellValue}>HONORABLE</Text>
                </View>
                 <View style={[styles.cell, styles.flex2]}>
                     <Text style={styles.cellLabel}>25. SEPARATION AUTHORITY</Text>
                     <Text style={styles.cellValue}>AR 635-200, CHAP 4</Text>
                </View>
            </View>
             <View style={styles.row}>
                <View style={[styles.cell, styles.flex1]}>
                     <Text style={styles.cellLabel}>26. SEPARATION CODE</Text>
                     <Text style={styles.cellValue}>JBK</Text>
                </View>
                 <View style={[styles.cell, styles.flex1]}>
                     <Text style={styles.cellLabel}>27. REENTRY CODE</Text>
                     <Text style={styles.cellValue}>RE-1</Text>
                </View>
                 <View style={[styles.cell, styles.flex2]}>
                     <Text style={styles.cellLabel}>28. NARRATIVE REASON</Text>
                     <Text style={styles.cellValue}>COMPLETION OF REQUIRED ACTIVE SERVICE</Text>
                </View>
            </View>
             <View style={[styles.row, { borderBottomWidth: 0 }]}>
                <View style={[styles.cell, styles.flex2, { borderBottomWidth: 0 }]}>
                     <Text style={styles.cellLabel}>29. DATES OF TIME LOST</Text>
                     <Text style={styles.cellValue}>NONE</Text>
                </View>
                 <View style={[styles.cell, styles.flex1, { borderBottomWidth: 0 }]}>
                     <Text style={styles.cellLabel}>30. MEMBER REQUESTS COPY 4</Text>
                     <Text style={styles.cellValue}>YES</Text>
                </View>
            </View>

        </View>

        {/* Footer */}
        <View style={styles.footer}>
             <View style={styles.signatureBlock}>
                  <Text style={styles.signatureText}>{data.firstName} {data.lastName}</Text>
                  <View style={styles.signatureLine}>
                      <Text style={styles.signatureLabel}>MEMBER SIGNATURE</Text>
                  </View>
             </View>
             <View style={[styles.signatureBlock, { alignItems: 'flex-end' }]}>
                  <Text style={[styles.signatureText, { paddingRight: 10 }]}>{data.authorizedOfficialName || 'GEN. A. Official'}</Text>
                  <View style={[styles.signatureLine, { width: '100%' }]}>
                       <Text style={[styles.signatureLabel, { textAlign: 'right' }]}>OFFICIAL AUTHORIZED SIGNATURE</Text>
                  </View>
             </View>
        </View>



        <Text style={styles.privacyAct}>
            AUTHORITY: 10 U.S.C. 1168. PRINCIPAL PURPOSE: To record transfer or discharge. ROUTINE USES: To the Dept of Veterans Affairs. DISCLOSURE: Voluntary; however, failure to provide info may result in non-issuance.
        </Text>

      </Page>
    </Document>
  );
};
