import React, { useState } from 'react';
import { Button } from '@/components/ui/Button';
import { Select } from '@/components/ui/Select';
import { Input } from '@/components/ui/Input';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/Dialog';
import { useTemplates } from '@/hooks/useTemplates';
import { TemplateType, Template } from '@/types/templates';
import { Save, FolderOpen, Trash2, Plus, Settings, Lock } from 'lucide-react';
import { Card, CardContent } from '@/components/ui/Card';
import { useRole } from '@/contexts/RoleContext';

interface TemplateManagerProps {
    type: TemplateType;
    onApply: (data: any) => void;
    currentData?: any; // Data to save when creating a new template
    label?: string; // e.g. "Institution"
}

export function TemplateManager({ type, onApply, currentData, label = 'Template' }: TemplateManagerProps) {
    const { templates, addTemplate, deleteTemplate, getTemplatesByType } = useTemplates();
    const availableTemplates = getTemplatesByType(type);
    const { isAdmin } = useRole();
    
    // States
    const [selectedId, setSelectedId] = useState<string>('');
    const [isSaveModalOpen, setIsSaveModalOpen] = useState(false);
    const [isManageModalOpen, setIsManageModalOpen] = useState(false);
    const [newTemplateName, setNewTemplateName] = useState('');

    const handleApply = () => {
        const template = availableTemplates.find(t => t.id === selectedId);
        if (template && template.data) {
            onApply(template.data);
        }
    };

    const handleSave = () => {
        if (!newTemplateName || !currentData) return;
        
        const newTemplate: Template = {
            id: `tpl-${Date.now()}`,
            name: newTemplateName,
            type: type as any, // TypeScript union matching
            createdAt: new Date().toISOString(),
            isLocked: false,
            data: currentData
        };

        addTemplate(newTemplate);
        setIsSaveModalOpen(false);
        setNewTemplateName('');
        setSelectedId(newTemplate.id); // Auto-select new
    };

    return (
        <div className="flex items-end gap-2 mb-4 p-4 border rounded-lg bg-slate-50 dark:bg-slate-900/50">
            <div className="flex-1 space-y-2">
                <label className="text-sm font-medium flex items-center gap-2">
                    <FolderOpen className="w-4 h-4" />
                    Load {label} Preset
                </label>
                <div className="flex gap-2">
                    <Select 
                        value={selectedId} 
                        onChange={(e) => setSelectedId(e.target.value)}
                        className="flex-1"
                    >
                        <option value="">Select a preset...</option>
                        {availableTemplates.map(t => (
                            <option key={t.id} value={t.id}>{t.name}</option>
                        ))}
                    </Select>
                    <Button onClick={handleApply} disabled={!selectedId} variant="secondary">
                        Apply
                    </Button>
                </div>
            </div>

            <div className="flex gap-2">
                 {isAdmin && (
                    <>
                        <Button 
                            variant="outline" 
                            title="Save current as template"
                            onClick={() => setIsSaveModalOpen(true)}
                            disabled={!currentData}
                        >
                            <Save className="w-4 h-4 mr-2" />
                            Save New
                        </Button>
                        <Button 
                            variant="ghost" 
                            size="icon"
                            title="Manage Templates"
                            onClick={() => setIsManageModalOpen(true)}
                        >
                            <Settings className="w-4 h-4" />
                        </Button>
                    </>
                 )}
            </div>

            {/* Save Modal */}
            <Dialog open={isSaveModalOpen} onOpenChange={setIsSaveModalOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Save New {label} Template</DialogTitle>
                    </DialogHeader>
                    <div className="py-4">
                        <label className="text-sm font-medium mb-2 block">Template Name</label>
                        <Input 
                            value={newTemplateName} 
                            onChange={(e) => setNewTemplateName(e.target.value)}
                            placeholder="e.g. University of Washington - Standard"
                        />
                    </div>
                    <DialogFooter>
                        <Button variant="outline" onClick={() => setIsSaveModalOpen(false)}>Cancel</Button>
                        <Button onClick={handleSave} disabled={!newTemplateName}>Save Template</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            {/* Manage Modal */}
            <Dialog open={isManageModalOpen} onOpenChange={setIsManageModalOpen}>
                <DialogContent className="max-w-xl">
                    <DialogHeader>
                        <DialogTitle>Manage {label} Templates</DialogTitle>
                    </DialogHeader>
                    <div className="max-h-[400px] overflow-y-auto space-y-2 py-4">
                        {availableTemplates.length === 0 && (
                            <p className="text-center text-muted-foreground p-4">No templates found.</p>
                        )}
                        {availableTemplates.map(t => (
                            <Card key={t.id}>
                                <CardContent className="flex items-center justify-between p-3">
                                    <div>
                                        <div className="font-medium">{t.name}</div>
                                        <div className="text-xs text-muted-foreground">Created: {new Date(t.createdAt).toLocaleDateString()}</div>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        {t.isLocked && <span className="text-xs bg-slate-100 px-2 py-1 rounded">Default</span>}
                                        {!t.isLocked && (
                                            <Button 
                                                variant="destructive" 
                                                size="sm" 
                                                onClick={() => deleteTemplate(t.id)}
                                            >
                                                <Trash2 className="w-4 h-4" />
                                            </Button>
                                        )}
                                    </div>
                                </CardContent>
                            </Card>
                        ))}
                    </div>
                     <DialogFooter>
                        <Button onClick={() => setIsManageModalOpen(false)}>Close</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>
        </div>
    );
}
