"use client";

import React, { useEffect, useRef } from 'react';

interface LogEntry {
  id: string;
  timestamp: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error';
}

interface SystemTerminalProps {
  logs: LogEntry[];
}

export const SystemTerminal = ({ logs }: SystemTerminalProps) => {
  const scrollRef = useRef<HTMLDivElement>(null);
  const [currentTime, setCurrentTime] = React.useState<string>('');

  // Auto-scroll to bottom on new logs
  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [logs]);

  // Client-side time to prevent hydration mismatch
  useEffect(() => {
    setCurrentTime(new Date().toLocaleTimeString('en-US', {hour12: false}));
    const interval = setInterval(() => {
       setCurrentTime(new Date().toLocaleTimeString('en-US', {hour12: false}));
    }, 1000);
    return () => clearInterval(interval);
  }, []);

  return (
    <div className="bg-black border border-emerald-900/50 rounded-lg overflow-hidden font-mono text-xs shadow-2xl flex flex-col h-full min-h-[150px]">
      {/* Terminal Header */}
      <div className="bg-emerald-950/30 border-b border-emerald-900/50 px-3 py-1 flex items-center justify-between">
        <div className="flex items-center space-x-2">
            <div className="w-2 h-2 rounded-full bg-red-500/50"></div>
            <div className="w-2 h-2 rounded-full bg-yellow-500/50"></div>
            <div className="w-2 h-2 rounded-full bg-green-500/50"></div>
        </div>
        <span className="text-emerald-700 font-bold tracking-widest text-[10px]">SYS_LOG.EXE</span>
      </div>

      {/* Terminal Content */}
      <div 
        ref={scrollRef}
        className="flex-1 p-3 overflow-y-auto space-y-1 scrollbar-thin scrollbar-thumb-emerald-900 scrollbar-track-transparent"
        style={{ maxHeight: '200px' }}
      >
        <div className="text-emerald-900/50 mb-2">Microsoft Windows [Version 10.0.19045.3693]<br/>(c) Microsoft Corporation. All rights reserved.</div>
        
        {logs.map((log) => (
          <div key={log.id} className="flex">
            <span className="text-emerald-700 mr-2 shrink-0">[{log.timestamp}]</span>
            <span className={`
                break-all
                ${log.type === 'error' ? 'text-red-500' : ''}
                ${log.type === 'warning' ? 'text-yellow-500' : ''}
                ${log.type === 'success' ? 'text-emerald-300 font-bold' : ''}
                ${log.type === 'info' ? 'text-emerald-500' : ''}
            `}>
              {log.type === 'success' ? '>> ' : '> '} {log.message}
            </span>
          </div>
        ))}
        
        <div className="flex animate-pulse">
            <span className="text-emerald-700 mr-2">[{currentTime}]</span>
            <span className="text-emerald-500">_</span>
        </div>
      </div>
    </div>
  );
};
