"use client";

import React, { useState, useEffect, useMemo } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/Card';
import { Label } from '@/components/ui/Label';
import { Input } from '@/components/ui/Input';
import { Select } from '@/components/ui/Select';
import { Button } from '@/components/ui/Button';
import { DateSelect } from '@/components/DateSelect';
import { 
  RefreshCw, Loader2, Download, FileText, User, BookOpen, Settings, 
  Shield, Camera, School, Trash2, Plus, List, CreditCard, GraduationCap,
  History, Save, Upload, RotateCcw, AlertTriangle, CheckCircle, AlertCircle
} from 'lucide-react';
import { pdf } from '@react-pdf/renderer';
import { StudentPacketPdf } from './StudentPacketPdf';
import { SystemTerminal } from './SystemTerminal';
import { ServiceRecordsIndex } from './ServiceRecordsIndex';
import { validateStudentData } from '@/utils/verificationValidation';
import { useServiceRecords } from '@/hooks/useServiceRecords';
import { ServiceRecord, StudentRecord } from '@/types/records';
import { TemplateManager } from './TemplateManager';
import { InstitutionTemplate, PacketPresetTemplate } from '@/types/templates';
import { BulkCourseImporter } from './BulkCourseImporter';
import { PdfGenerationModal } from './PdfGenerationModal';
import {
  StudentVerificationData,
  DEFAULT_INSTITUTIONS,
  CREDENTIAL_TYPES,
  Course,
  ReceiptItem,
  TranscriptEntry,
  PdfSettings,
  IdCardSettings
} from '@/types/student-verification';

// Log definition
interface LogEntry {
  id: string;
  timestamp: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error';
}

const INITIAL_DATA: Partial<StudentVerificationData> = {
  includeIdCard: true,
  includeEnrollmentLetter: true,
  includeClassSchedule: true,
  includeTuitionReceipt: true,
  includeTranscript: true,
  includeRegistrationReceipt: true,
  includeGoodStandingLetter: false,
  includeFinancialStatement: false,
  courses: [],
  tuitionItems: [],
  registrationItems: [],
  transcriptEntries: [],
};

const FIRST_NAMES = [
  "James", "Mary", "Robert", "Patricia", "John", "Jennifer", "Michael", "Linda", 
  "David", "Elizabeth", "William", "Barbara", "Richard", "Susan", "Joseph", 
  "Jessica", "Thomas", "Sarah", "Charles", "Karen", "Christopher", "Nancy",
  "Daniel", "Lisa", "Matthew", "Betty", "Anthony", "Margaret", "Mark", "Sandra",
  "Paul", "Steven", "Andrew", "Kenneth", "Joshua", "Kevin", "Brian", "George",
  "Edward", "Ronald", "Timothy", "Jason", "Jeffrey", "Ryan", "Jacob", "Gary",
  "Nicholas", "Eric", "Jonathan", "Stephen", "Larry", "Justin", "Scott", "Brandon",
  "Benjamin", "Samuel", "Gregory", "Frank", "Alexander", "Raymond", "Patrick",
  "Jack", "Dennis", "Jerry", "Tyler", "Aaron", "Jose", "Adam", "Henry", "Nathan",
  "Douglas", "Zachary", "Peter", "Kyle", "Walter", "Ethan", "Jeremy", "Harold",
  "Keith", "Christian", "Roger", "Terry", "Gerald", "Sean", "Austin", "Carl",
  "Arthur", "Lawrence", "Dylan", "Jesse", "Jordan", "Billy", "Joe", "Bruce",
  "Gabriel", "Logan", "Albert", "Willie", "Alan", "Juan", "Wayne", "Elijah",
  "Randy", "Roy", "Vincent", "Ralph", "Eugene", "Russell", "Bobby", "Mason",
  "Philip", "Louis"
];

const LAST_NAMES = [
  "Smith", "Johnson", "Williams", "Brown", "Jones", "Garcia", "Miller", "Davis",
  "Rodriguez", "Martinez", "Hernandez", "Lopez", "Gonzalez", "Wilson", "Anderson",
  "Thomas", "Taylor", "Moore", "Jackson", "Martin", "Lee", "Perez", "Thompson",
  "White", "Harris", "Sanchez", "Clark", "Ramirez", "Lewis", "Robinson",
  "Walker", "Young", "Allen", "King", "Wright", "Scott", "Torres", "Nguyen",
  "Hill", "Flores", "Green", "Adams", "Nelson", "Baker", "Hall", "Rivera",
  "Campbell", "Mitchell", "Carter", "Roberts", "Gomez", "Phillips", "Evans",
  "Turner", "Diaz", "Parker", "Cruz", "Edwards", "Collins", "Reyes", "Stewart",
  "Morris", "Morales", "Murphy", "Cook", "Rogers", "Gutierrez", "Ortiz", "Morgan",
  "Cooper", "Peterson", "Bailey", "Reed", "Kelly", "Howard", "Ramos", "Kim",
  "Cox", "Ward", "Richardson", "Watson", "Brooks", "Chavez", "Wood", "James",
  "Bennett", "Gray", "Mendoza", "Ruiz", "Hughes", "Price", "Alvarez", "Castillo",
  "Sanders", "Patel", "Myers", "Long", "Ross", "Foster", "Jimenez", "Powell",
  "Jenkins", "Perry", "Russell", "Sullivan", "Bell", "Coleman", "Butler", "Henderson",
  "Barnes", "Gonzales", "Fisher", "Vasquez", "Simmons", "Romero", "Jordan", 
  "Patterson", "Alexander", "Hamilton", "Graham", "Reynolds", "Griffin", "Wallace"
];

const PROGRAMS = [
  "Computer Science", "Business Administration", "Psychology", "Nursing", "Biology",
  "Mechanical Engineering", "Education", "Communications", "Economics", "Political Science",
  "Electrical Engineering", "English Literature", "Accounting", "Sociology", "Art History",
  "Marketing", "Environmental Science", "Chemistry", "Mathematics", "Physics",
  "Civil Engineering", "Chemical Engineering", "Industrial Engineering", "Aerospace Engineering", 
  "Biomedical Engineering", "Architecture", "Graphic Design", "Interior Design", "Urban Planning", 
  "Fine Arts", "Music Performance", "Film Studies", "Journalism", "Public Relations", 
  "Advertising", "History", "Philosophy", "Anthropology", "Geography", "Geology", 
  "Astronomy", "Statistics", "Actuarial Science", "Data Science", "Cyber Security", 
  "Information Technology", "Software Engineering", "Game Design", "Nutrition", "Public Health", 
  "Health Administration", "Kinesiology", "Sports Management", "Hospitality Management", 
  "Culinary Arts", "Fashion Design", "Textile Engineering", "Agriculture", "Horticulture", 
  "Forestry", "Marine Biology", "Oceanography", "Linguistics", "International Relations", 
  "Social Work", "Criminal Justice", "Criminology", "Forensic Science", "Law", 
  "Theology", "Library Science", "Veterinary Medicine", "Pharmacy", "Dentistry",
  "Physiotherapy", "Occupational Therapy", "Radiography", "Paramedicine", "Midwifery",
  "Genetics", "Microbiology", "Biochemistry", "Biotechnology", "Neuroscience", 
  "Ecology", "Zoology", "Botany", "Meteorology", "Materials Science", "Nanotechnology",
  "Robotics", "Mechatronics", "Petroleum Engineering", "Mining Engineering", "Nuclear Engineering"
];

const INITIAL_PDF_SETTINGS: PdfSettings = {
  paperSize: 'LETTER',
  margins: 'NORMAL',
  includePageNumbers: true,
  showPacketId: true,
  includeSectionDividers: true,
  includeTableOfContents: true,
};

const INITIAL_ID_SETTINGS: IdCardSettings = {
  theme: 'LIGHT',
  barcodeType: 'QR',
  expirationMode: 'AUTO',
};

const DEMO_COURSES: Course[] = [
  { id: '1', code: 'CS101', title: 'Intro to Computer Science', instructor: 'Dr. Smith', days: 'MWF', time: '10:00 - 11:30', room: 'Bldg A - 101', credits: 3 },
  { id: '2', code: 'MATH201', title: 'Calculus II', instructor: 'Prof. Johnson', days: 'TTh', time: '13:00 - 14:30', room: 'Bldg B - 204', credits: 4 },
  { id: '3', code: 'ENG102', title: 'Technical Writing', instructor: 'Ms. Davis', days: 'Fri', time: '09:00 - 12:00', room: 'Lib - 305', credits: 3 },
];

const DEMO_TUITION_ITEMS: ReceiptItem[] = [
  { id: '1', description: 'Tuition (12 Credits)', amount: 4500 },
  { id: '2', description: 'Technology Fee', amount: 150 },
  { id: '3', description: 'Student Activity Fee', amount: 75 },
  { id: '4', description: 'Campus Health Fee', amount: 120 },
];

const DEMO_TRANSCRIPT: TranscriptEntry[] = [
  { id: '1', term: 'Fall 2024', course: 'CS101', credits: 3, grade: 'A' },
  { id: '2', term: 'Fall 2024', course: 'MATH101', credits: 4, grade: 'B+' },
  { id: '3', term: 'Spring 2025', course: 'CS102', credits: 3, grade: 'A-' },
  { id: '4', term: 'Spring 2025', course: 'PHYS101', credits: 4, grade: 'B' },
];

export function StudentVerificationForm() {
  const [formData, setFormData] = useState<Partial<StudentVerificationData>>({});
  const [isGenerating, setIsGenerating] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [successMessage, setSuccessMessage] = useState<string | null>(null);
  const [errorMessage, setErrorMessage] = useState<string | null>(null);
  const [logs, setLogs] = useState<LogEntry[]>([]);
  const [activeTab, setActiveTab] = useState<'courses' | 'transcript' | 'tuition' | 'registration'>('courses');
  const [mainTab, setMainTab] = useState<'form' | 'history'>('form');

  // New States
  const [pdfSettings, setPdfSettings] = useState<PdfSettings>(INITIAL_PDF_SETTINGS);
  const [idSettings, setIdSettings] = useState<IdCardSettings>(INITIAL_ID_SETTINGS);
  const [forceGenerate, setForceGenerate] = useState(false);
  const [isBulkImportOpen, setIsBulkImportOpen] = useState(false);
  const [isPdfModalOpen, setIsPdfModalOpen] = useState(false);

  const { records, addRecord, deleteRecord, searchRecords } = useServiceRecords();

  // Load session on mount
  useEffect(() => {
    const saved = localStorage.getItem('student_verify_session');
    if (saved) {
      try {
        const parsed = JSON.parse(saved);
        // Date strings back to Date objects
        if (parsed.dob) parsed.dob = new Date(parsed.dob);
        if (parsed.termStartDate) parsed.termStartDate = new Date(parsed.termStartDate);
        if (parsed.termEndDate) parsed.termEndDate = new Date(parsed.termEndDate);
        if (parsed.expectedGraduationDate) parsed.expectedGraduationDate = new Date(parsed.expectedGraduationDate);
        setFormData(parsed);
        addLog('SESSION RESTORED FROM LOCAL STORAGE.', 'info');
      } catch (e) {
        console.error("Failed to parse session", e);
      }
    } else {
        setFormData(INITIAL_DATA);
    }
  }, []);

  // Save session on change
  useEffect(() => {
    if (Object.keys(formData).length > 0) {
        try {
            // Create a safe copy for storage, excluding large images
            const storageData = { ...formData };
            delete storageData.studentPhoto;
            delete storageData.institutionSeal;

            localStorage.setItem('student_verify_session', JSON.stringify(storageData));
        } catch (e) {
            console.warn("Failed to save session to localStorage (Quota Exceeded?)", e);
        }
    }
  }, [formData]);

  // Validation Check
  const validationResult = useMemo(() => validateStudentData(formData as StudentVerificationData), [formData]);
  const canGenerate = validationResult.isValid || forceGenerate;

  const addLog = (message: string, type: LogEntry['type'] = 'info') => {
      const now = new Date();
      const timeString = now.toLocaleTimeString('en-US', { hour12: false });
      const newLog: LogEntry = {
          id: Math.random().toString(36).substr(2, 9),
          timestamp: timeString,
          message,
          type
      };
      setLogs(prev => [...prev.slice(-49), newLog]); // Keep last 50
  };

  const handleAutofill = () => {
    setIsGenerating(true);
    setSuccessMessage(null);
    setErrorMessage(null);
    addLog('INITIATING SECURE DATABASE QUERY...', 'info');

    setTimeout(() => {
      const now = new Date();
      
      // Helper for random selection
      const pick = <T,>(arr: T[] | readonly T[]): T => arr[Math.floor(Math.random() * arr.length)];
      
      // General Randomizers
      const firstName = pick(FIRST_NAMES);
      const lastName = pick(LAST_NAMES);
      const institution = pick(DEFAULT_INSTITUTIONS);
      const program = pick(PROGRAMS);
      const credType = pick(CREDENTIAL_TYPES);
      
      // Email Generation Logic
      const domains = ['student.edu', 'university.ac.uk', 'college.edu', 'campus.net', 'academy.org', 'inst.edu'];
      const domain = pick(domains);
      
      const randNum = Math.floor(Math.random() * 899) + 100; // 100-999
      let baseName = '';
      const emailType = Math.random();
      
      if (emailType < 0.33) {
          baseName = `${firstName.toLowerCase()}.${lastName.toLowerCase()}`;
      } else if (emailType < 0.66) {
          baseName = `${firstName.charAt(0).toLowerCase()}${lastName.toLowerCase()}`;
      } else {
          baseName = `${firstName.toLowerCase()}${lastName.toLowerCase()}`;
      }
      const email = `${baseName}${randNum}@${domain}`;

      // Age between 19 and 28
      const age = Math.floor(Math.random() * 9) + 19;
      const dob = new Date(now.getFullYear() - age, Math.floor(Math.random() * 12), Math.floor(Math.random() * 28) + 1);

      // Randomize Term & Dates based on CURRENT date for coherence
      const currentYear = now.getFullYear();
      const currentMonth = now.getMonth(); // 0-11

      let currentTerm = '';
      let termStartDate = new Date();
      let termEndDate = new Date();
      
      // Determine efficient active term
      if (currentMonth >= 0 && currentMonth <= 4) { // Jan - May (Spring)
          currentTerm = 'Spring';
          // Start early Jan
          termStartDate = new Date(currentYear, 0, 10 + Math.floor(Math.random() * 10)); 
          // End mid May
          termEndDate = new Date(currentYear, 4, 15 + Math.floor(Math.random() * 10));
      } else if (currentMonth >= 5 && currentMonth <= 7) { // Jun - Aug (Summer)
          currentTerm = 'Summer';
          termStartDate = new Date(currentYear, 5, 1 + Math.floor(Math.random() * 10));
          termEndDate = new Date(currentYear, 7, 15 + Math.floor(Math.random() * 10));
      } else { // Sep - Dec (Fall)
          currentTerm = 'Fall';
          termStartDate = new Date(currentYear, 8, 1 + Math.floor(Math.random() * 10));
          termEndDate = new Date(currentYear, 11, 15 + Math.floor(Math.random() * 10));
      }

      // Graduation 1-4 years out from NOW
      const gradYearOffset = Math.floor(Math.random() * 3) + 1; // 1-3 years
      const expectedGraduationDate = new Date(currentYear + gradYearOffset, 4, 15 + Math.floor(Math.random() * 15));

      // Registrar
      const regFirst = pick(FIRST_NAMES);
      const regLast = pick(LAST_NAMES);
      const regPhone = `555-${Math.floor(Math.random() * 899) + 100}-${Math.floor(Math.random() * 8999) + 1000}`;

      // Program-based Course Logic
      const PROGRAM_COURSES: Record<string, any[]> = {
        'Computer Science': [
          { code: 'CS301', title: 'Data Structures', credits: 4 },
          { code: 'CS305', title: 'Algorithms', credits: 3 },
          { code: 'MATH201', title: 'Linear Algebra', credits: 3 },
          { code: 'CS401', title: 'Operating Systems', credits: 4 }
        ],
        'Nursing': [
          { code: 'NURS201', title: 'Human Anatomy', credits: 4 },
          { code: 'NURS205', title: 'Pharmacology I', credits: 3 },
          { code: 'BIO101', title: 'Microbiology', credits: 3 },
          { code: 'PSY101', title: 'Intro to Psychology', credits: 3 }
        ],
        'Business Administration': [
          { code: 'MGT101', title: 'Principles of Management', credits: 3 },
          { code: 'ACC201', title: 'Financial Accounting', credits: 4 },
          { code: 'ECO101', title: 'Microeconomics', credits: 3 },
          { code: 'MKT301', title: 'Marketing Strategy', credits: 3 }
        ],
        'Psychology': [
          { code: 'PSY201', title: 'Abnormal Psychology', credits: 3 },
          { code: 'PSY205', title: 'Research Methods', credits: 4 },
          { code: 'SOC101', title: 'Intro to Sociology', credits: 3 },
          { code: 'BIO102', title: 'Human Biology', credits: 3 }
        ]
      };

      const defaultCourses = [
        { code: 'GEN101', title: 'General Education I', credits: 3 },
        { code: 'GEN102', title: 'Critical Thinking', credits: 3 },
        { code: 'EL101', title: 'Elective I', credits: 3 },
        { code: 'EL102', title: 'Elective II', credits: 3 }
      ];

      // Select courses based on program or default
      const selectedCoursesRaw = PROGRAM_COURSES[program] || defaultCourses;
      
      // Generate unique IDs and schedule details for courses
      const generatedCourses = selectedCoursesRaw.map((c, i) => ({
        id: Math.random().toString(36).substr(2, 9),
        code: c.code,
        title: c.title,
        credits: c.credits,
        instructor: `${pick(LAST_NAMES)}, ${pick(FIRST_NAMES).charAt(0)}.`,
        days: ['Mon/Wed', 'Tue/Thu', 'Fri'][Math.floor(Math.random() * 3)],
        time: ['09:00 AM', '11:00 AM', '01:00 PM', '03:00 PM'][Math.floor(Math.random() * 4)],
        room: `Bldg ${String.fromCharCode(65 + Math.floor(Math.random() * 5))}-${Math.floor(Math.random() * 300)}`
      }));

      // Generate Transcript (Previous Terms based on current)
      const transcriptEntries = [];
      const pastTerms = [];
      // If Spring 2026 -> Fall 2025, Spring 2025, Fall 2024
      // If Fall 2026 -> Summer 2026, Spring 2026, Fall 2025
      
      let tYear = currentYear;
      if (currentTerm === 'Spring') {
          pastTerms.push(`Fall ${tYear-1}`);
          pastTerms.push(`Spring ${tYear-1}`);
          pastTerms.push(`Fall ${tYear-2}`);
      } else if (currentTerm === 'Summer') {
          pastTerms.push(`Spring ${tYear}`);
          pastTerms.push(`Fall ${tYear-1}`);
          pastTerms.push(`Spring ${tYear-1}`);
      } else { // Fall
          pastTerms.push(`Spring ${tYear}`);
          pastTerms.push(`Fall ${tYear-1}`);
          pastTerms.push(`Spring ${tYear-1}`);
      }

      let transId = 1;
      for (const t of pastTerms) {
          transcriptEntries.push({
              id: (transId++).toString(),
              term: t,
              course: 'Generated Course ' + transId, // Simplified for history
              credits: 3,
              grade: ['A', 'A-', 'B+', 'B'][Math.floor(Math.random() * 4)]
          });
      }

      // Generate Tuition
      const tuitionItems = [
          { id: '1', description: 'Tuition Fee - ' + program, amount: Math.floor(Math.random() * 2000) + 3000 },
          { id: '2', description: 'Student Activities Fee', amount: 150 },
          { id: '3', description: 'Technology Fee', amount: 200 },
          { id: '4', description: 'Health Service Fee', amount: 180 }
      ];

      const newData: Partial<StudentVerificationData> = {
        firstName: firstName,
        lastName: lastName,
        dob: dob,
        email: email,
        studentId: Math.floor(10000000 + Math.random() * 90000000).toString(),
        institutionName: institution,
        programName: program,
        credentialType: credType,
        enrollmentStatus: 'Active',
        termName: `${currentTerm} ${currentYear}`,
        termStartDate: termStartDate,
        termEndDate: termEndDate,
        creditsThisTerm: generatedCourses.reduce((sum, c) => sum + c.credits, 0),
        expectedGraduationDate: expectedGraduationDate,
        registrarContact: `${regFirst} ${regLast} | ${regPhone}`,

        includeIdCard: true,
        includeEnrollmentLetter: true,
        includeClassSchedule: true,
        includeTuitionReceipt: true,
        includeTranscript: true,
        includeRegistrationReceipt: true,
        includeGoodStandingLetter: Math.random() > 0.5,
        includeFinancialStatement: false,

        courses: generatedCourses,
        tuitionItems: tuitionItems,
        registrationItems: [{ id: '1', description: 'Registration Fee', amount: 50 }],
        transcriptEntries: transcriptEntries,
      };

      setFormData(prev => ({...prev, ...newData}));
      setErrors({});
      setIsGenerating(false);
      addLog(`STUDENT RECORD RETRIEVED: ${lastName.toUpperCase()}, ${firstName.toUpperCase()}`, 'success');
    }, 600);
  };

  const handleClear = () => {
    if (confirm("Are you sure you want to clear all data? This cannot be undone.")) {
        setFormData(INITIAL_DATA);
        setErrors({});
        setSuccessMessage(null);
        setErrorMessage(null);
        localStorage.removeItem('student_verify_session');
        addLog('FORM RESET. READY FOR NEW INPUT.', 'warning');
    }
  };

  const handleChange = (field: keyof StudentVerificationData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    if (errors[field]) {
      setErrors(prev => {
        const newErrors = { ...prev };
        delete newErrors[field];
        return newErrors;
      });
    }
  };

  // Array item handlers
  const handleAddItem = (type: 'courses' | 'transcript' | 'tuition' | 'registration') => {
      const id = Math.random().toString(36).substr(2, 9);
      setFormData(prev => {
          const newData = { ...prev };
          switch(type) {
              case 'courses':
                  newData.courses = [...(prev.courses || []), { id, code: '', title: '', instructor: '', days: '', time: '', room: '', credits: 3 }];
                  break;
              case 'transcript':
                  newData.transcriptEntries = [...(prev.transcriptEntries || []), { id, term: prev.termName || '', course: '', credits: 3, grade: 'A' }];
                  break;
              case 'tuition':
                  newData.tuitionItems = [...(prev.tuitionItems || []), { id, description: '', amount: 0 }];
                  break;
              case 'registration':
                  newData.registrationItems = [...(prev.registrationItems || []), { id, description: '', amount: 0 }];
                  break;
          }
          return newData;
      });
  };

  const handleRemoveItem = (type: 'courses' | 'transcript' | 'tuition' | 'registration', id: string) => {
      setFormData(prev => {
          const newData = { ...prev };
          switch(type) {
              case 'courses':
                  newData.courses = prev.courses?.filter(c => c.id !== id);
                  break;
              case 'transcript':
                  newData.transcriptEntries = prev.transcriptEntries?.filter(t => t.id !== id);
                  break;
              case 'tuition':
                  newData.tuitionItems = prev.tuitionItems?.filter(t => t.id !== id);
                  break;
              case 'registration':
                  newData.registrationItems = prev.registrationItems?.filter(t => t.id !== id);
                  break;
          }
          return newData;
      });
  };

  const handleUpdateItem = (type: 'courses' | 'transcript' | 'tuition' | 'registration', id: string, field: string, value: any) => {
      setFormData(prev => {
          const newData = { ...prev };
          const update = (arr: any[]) => arr.map(item => item.id === id ? { ...item, [field]: value } : item);

          switch(type) {
              case 'courses':
                  newData.courses = update(prev.courses || []);
                  break;
              case 'transcript':
                  newData.transcriptEntries = update(prev.transcriptEntries || []);
                  break;
              case 'tuition':
                  newData.tuitionItems = update(prev.tuitionItems || []);
                  break;
              case 'registration':
                  newData.registrationItems = update(prev.registrationItems || []);
                  break;
          }
          return newData;
      });
  };

  const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>, field: 'institutionSeal' | 'studentPhoto') => {
      const file = event.target.files?.[0];
      if (file) {
          addLog(`UPLOADING ASSET: ${file.name.toUpperCase()}...`, 'info');
          const reader = new FileReader();
          reader.onloadend = () => {
              handleChange(field, reader.result as string);
              addLog(`ASSET UPLOAD COMPLETE: ${field.toUpperCase()}`, 'success');
          };
          reader.readAsDataURL(file);
      }
  };

  const handleDuplicateHistory = (record: ServiceRecord) => {
    if (record.type !== 'STUDENT') {
        alert("Only Student records can be loaded here."); // or handle gracefully
        return;
    }
    const studentRecord = record as StudentRecord;

    if (confirm(`Load packet "${record.systemRef}"? This will replace your current form data.`)) {
        const restored = JSON.parse(JSON.stringify(studentRecord.data));

        // Hydrate dates
        if (restored.dob) restored.dob = new Date(restored.dob);
        if (restored.termStartDate) restored.termStartDate = new Date(restored.termStartDate);
        if (restored.termEndDate) restored.termEndDate = new Date(restored.termEndDate);
        if (restored.expectedGraduationDate) restored.expectedGraduationDate = new Date(restored.expectedGraduationDate);

        setFormData(restored);
        setPdfSettings(studentRecord.pdfSettings);
        setMainTab('form');
        addLog(`SESSION RESTORED FROM HISTORY: ${record.systemRef}`, 'success');
    }
  };

  const handleImportJSON = () => {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = 'application/json';
    input.onchange = (e) => {
      const file = (e.target as HTMLInputElement).files?.[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = (e) => {
          try {
            const imported = JSON.parse(e.target?.result as string);
            // Basic validation check
            if (imported.firstName || imported.studentId) {
                // Handle Date conversions
                if (imported.dob) imported.dob = new Date(imported.dob);
                if (imported.termStartDate) imported.termStartDate = new Date(imported.termStartDate);
                if (imported.termEndDate) imported.termEndDate = new Date(imported.termEndDate);

                setFormData(imported);
                addLog('SESSION IMPORTED SUCCESSFULLY', 'success');
            } else {
                addLog('IMPORT FAILED: INVALID DATA FORMAT', 'error');
            }
          } catch (err) {
            addLog('IMPORT FAILED: JSON PARSE ERROR', 'error');
          }
        };
        reader.readAsText(file);
      }
    };
    input.click();
  };

  const handleExportJSON = () => {
      const dataStr = JSON.stringify(formData, null, 2);
      const blob = new Blob([dataStr], { type: "application/json" });
      const url = URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      link.download = `student_packet_${formData.studentId || 'draft'}.json`;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      addLog('SESSION EXPORTED TO JSON', 'success');
  };

  const handleDownloadClick = () => {
     if (!forceGenerate && !validationResult.isValid) {
        addLog('VALIDATION FAILED. CANNOT GENERATE.', 'error');
        return;
     }
     setIsPdfModalOpen(true);
  };

  const executeDownload = async () => {
    setIsGenerating(true);
    setSuccessMessage(null);
    setErrorMessage(null);

    // Create generation timestamp & ref
    const timestamp = new Date();
    const sysRef = `2026 GK STU ${Math.floor(Math.random() * 900 + 100)}`;

    try {
        const blob = await pdf(
            <StudentPacketPdf
                data={formData}
                settings={pdfSettings}
                idSettings={idSettings}
                systemRef={sysRef}
            />
        ).toBlob();

        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        const filename = `Student_Verification_Packet_${formData.lastName || 'Unknown'}_${sysRef.replace(/ /g,'_')}.pdf`;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        link.remove();
        URL.revokeObjectURL(url);

        // Save to History (Unified)
        // Strip heavy assets for storage
        const safeHistoryData = { ...formData };
        delete safeHistoryData.studentPhoto;
        delete safeHistoryData.institutionSeal;

        const newRecord: StudentRecord = {
          id: Math.random().toString(36).substr(2, 9),
          type: 'STUDENT',
          systemRef: sysRef,
          generatedAt: timestamp.toISOString(),
          status: 'GENERATED',
          details: {
            name: `${formData.firstName} ${formData.lastName}`,
            institution: formData.institutionName || 'Unknown',
            term: formData.termName || 'Unknown',
          },
          data: JSON.parse(JSON.stringify(safeHistoryData)), 
          pdfSettings: { ...pdfSettings }
        };
        addRecord(newRecord);

        setIsGenerating(false);
        setSuccessMessage('Packet generated successfully.');
        addLog(`PACKET GENERATED: ${sysRef}`, 'success');

    } catch (err: any) {
        console.error('PDF Generation Error:', err);
        setIsGenerating(false);
        addLog(`SYSTEM ERROR: ${err.message || 'UNKNOWN FATAL ERROR'}`, 'error');
        setErrorMessage(`System Error: ${err.message || 'Failed to generate packet'}`);
    }
  };

  const handleApplyPacketTemplate = (data: PacketPresetTemplate['data']) => {
      const updates: Partial<StudentVerificationData> = {};

      // Reset all document toggles first
      const docKeys = [
          'includeIdCard', 'includeEnrollmentLetter', 'includeClassSchedule',
          'includeTuitionReceipt', 'includeTranscript', 'includeRegistrationReceipt',
          'includeGoodStandingLetter', 'includeFinancialStatement'
      ];
      docKeys.forEach(k => updates[k as keyof StudentVerificationData] = false);

      // Enable included
      data.includedDocuments.forEach(docId => {
          // simple mapping if ids match keys, or map them:
          // For this specific implementation, let's assume the template stores the full key name for simplicity,
          // or we map 'transcript' -> 'includeTranscript' etc.
          // Let's try direct key matching first.
          if (docKeys.includes(docId)) {
               updates[docId as keyof StudentVerificationData] = true;
          }
      });

      if (data.programName) updates.programName = data.programName;

      setFormData(prev => ({ ...prev, ...updates }));

      if (data.pdfSettings) {
          setPdfSettings(prev => ({ ...prev, ...data.pdfSettings }));
      }

      addLog('PACKET TEMPLATE APPLIED', 'success');
  };

  const handleApplyInstitutionTemplate = (data: InstitutionTemplate['data']) => {
      const updates: Partial<StudentVerificationData> = {
          institutionName: data.name
      };

      // Basic mapping for registrar contact since form uses a single string currently
      if (data.registrarName) {
           updates.registrarContact = `${data.registrarName} ${data.registrarTitle ? `(${data.registrarTitle})` : ''}`;
      }

      setFormData(prev => ({ ...prev, ...updates }));
      addLog(`INSTITUTION TEMPLATE APPLIED: ${data.name.toUpperCase()}`, 'success');
  };

  const handleBulkImport = (courses: Course[]) => {
      setFormData(prev => ({
          ...prev,
          courses: [...(prev.courses || []), ...courses]
      }));
      addLog(`BULK IMPORTED ${courses.length} COURSES`, 'success');
  };

  const handleSmartFillTranscript = () => {
      if (!formData.courses?.length) {
          alert("No courses available to generate transcript from.");
          return;
      }

      const newEntries: TranscriptEntry[] = formData.courses.map(c => ({
          id: Math.random().toString(36).substr(2, 9),
          term: formData.termName || 'Current Term',
          course: c.code,
          credits: c.credits,
          grade: 'IP' // In Progress default
      }));

      setFormData(prev => ({
          ...prev,
          transcriptEntries: [...(prev.transcriptEntries || []), ...newEntries]
      }));
      addLog(`GENERATED ${newEntries.length} TRANSCRIPT ENTRIES`, 'success');
  };

  const handleCalculateFees = () => {
      const COST_PER_CREDIT = 350; // Example default
      const TECH_FEE = 150;
      const REG_FEE = 50;

      const totalCredits = formData.courses?.reduce((sum, c) => sum + (c.credits || 0), 0) || 0;
      const tuitionAmt = totalCredits * COST_PER_CREDIT;

      // Tuition Items
      const newTuition: ReceiptItem[] = [
          { id: Math.random().toString(36).substr(2, 9), description: `Tuition (${totalCredits} Credits @ $${COST_PER_CREDIT})`, amount: tuitionAmt },
          { id: Math.random().toString(36).substr(2, 9), description: 'Technology Fee', amount: TECH_FEE },
          { id: Math.random().toString(36).substr(2, 9), description: 'Campus Activity Fee', amount: 75 },
      ];

      // Registration Items
      const newReg: ReceiptItem[] = [
          { id: Math.random().toString(36).substr(2, 9), description: 'Semester Registration', amount: REG_FEE },
          { id: Math.random().toString(36).substr(2, 9), description: 'ID Card Issuance', amount: 25 },
      ];

      setFormData(prev => ({
          ...prev,
          tuitionItems: newTuition,
          registrationItems: newReg
      }));
      addLog(`TUITION & FEES ASSESSMENT UPDATED: $${tuitionAmt + TECH_FEE + 75 + REG_FEE + 25}`, 'success');
  };

  // Helper to extract current packet state for saving
  const getCurrentPacketState = (): PacketPresetTemplate['data'] => {
      const includedDocuments: string[] = [];
       const docKeys = [
          'includeIdCard', 'includeEnrollmentLetter', 'includeClassSchedule',
          'includeTuitionReceipt', 'includeTranscript', 'includeRegistrationReceipt',
          'includeGoodStandingLetter', 'includeFinancialStatement'
      ];
      docKeys.forEach(k => {
          if (formData[k as keyof StudentVerificationData]) includedDocuments.push(k);
      });

      return {
          programName: formData.programName,
          includedDocuments,
          pdfSettings
      };
  };

  const getCurrentInstitutionState = (): InstitutionTemplate['data'] => {
      return {
          name: formData.institutionName || 'New Institution',
          // Parse registrar if possible, or leave blank for now as form doesn't support structured registrar
          registrarName: formData.registrarContact?.split('|')[0]?.trim(),
      };
  };

  return (
    <div className="space-y-6">

      {/* Top Action Bar */}
      <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4 bg-white dark:bg-slate-900 p-4 rounded-lg border border-slate-200 dark:border-slate-800 shadow-sm">
        <div>
            <h2 className="text-lg font-bold text-slate-800 dark:text-slate-100 flex items-center">
                <School className="w-5 h-5 mr-2 text-indigo-600" />
                ACADEMIC VERIFICATION CONTROL PANEL
            </h2>
            <p className="text-xs text-slate-500 font-mono mt-1" suppressHydrationWarning>SYSTEM REF: 2026-GK-STU-...</p>
        </div>
        <div className="flex space-x-3">
             <Button variant={mainTab === 'form' ? 'secondary' : 'ghost'} onClick={() => setMainTab('form')} size="sm" className="text-xs">
                <FileText className="mr-2 h-3 w-3" /> EDITOR
             </Button>
             <Button variant={mainTab === 'history' ? 'secondary' : 'ghost'} onClick={() => setMainTab('history')} size="sm" className="text-xs">
                <History className="mr-2 h-3 w-3" /> RECORDS ({records.length})
             </Button>
            <div className="h-6 w-px bg-slate-700 mx-2 hidden sm:block"></div>
             <Button variant="outline" onClick={handleAutofill} disabled={isGenerating} size="sm" className="text-xs border-slate-300 dark:border-slate-700">
                <RotateCcw className="mr-2 h-3 w-3" /> AUTO-POPULATE
             </Button>
             <Button variant="outline" onClick={handleClear} size="sm" className="text-xs border-slate-300 dark:border-slate-700">
                <Trash2 className="mr-2 h-3 w-3" /> CLEAR
             </Button>
            <Button onClick={handleDownloadClick} disabled={isGenerating} size="sm" className="bg-slate-800 hover:bg-slate-700 text-white text-xs">
                <RefreshCw className={`mr-2 h-3 w-3 ${isGenerating ? 'animate-spin' : ''}`} />
                PREVIEW & GENERATE
            </Button>

        </div>
      </div>

      <BulkCourseImporter 
         isOpen={isBulkImportOpen}
         onClose={() => setIsBulkImportOpen(false)}
         onImport={handleBulkImport}
      />
      <PdfGenerationModal 
         isOpen={isPdfModalOpen}
         onClose={() => setIsPdfModalOpen(false)}
         onComplete={executeDownload}
         studentName={`${formData.firstName} ${formData.lastName}`}
      />

      {mainTab === 'history' ? (
          <ServiceRecordsIndex
            records={records}
            onSearch={searchRecords}
            onLoadDraft={handleDuplicateHistory} // Loads logic
            onDelete={deleteRecord}
          />
      ) : (
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            
            {/* Left Col: Form Inputs */}
            <div className="lg:col-span-2 space-y-6">
                
                {/* GLOBAL PACKET TEMPLATE MANAGER */}
                <TemplateManager 
                    type="PACKET_PRESET" 
                    label="Packet Preset"
                    onApply={handleApplyPacketTemplate}
                    currentData={getCurrentPacketState()}
                />

                {/* VALIDATION WARNING */}
                {!validationResult.isValid && (
                   <div className="bg-amber-500/10 border border-amber-500/30 rounded p-3 flex items-start space-x-3">
                       <AlertTriangle className="text-amber-500 w-5 h-5 mt-0.5 flex-shrink-0" />
                       <div>
                           <h4 className="text-xs font-bold text-amber-500 uppercase mb-1">Incomplete Data</h4>
                           <ul className="text-[10px] text-amber-400/80 space-y-0.5 list-disc list-inside">
                               {validationResult.errors.map((err, i) => <li key={i}>{err}</li>)}
                           </ul>
                       </div>
                   </div>
                )}

                {/* Identity Card */}
                <Card className="border-t-4 border-t-indigo-500 shadow-sm">
                    <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                        <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                            <User className="w-4 h-4 mr-2" /> 
                            Identity Information
                        </CardTitle>
                    </CardHeader>
                    <CardContent className="pt-6 grid gap-6 sm:grid-cols-2">
                         <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold" required>First Name</Label>
                            <Input 
                                value={formData.firstName || ''} 
                                onChange={(e) => handleChange('firstName', e.target.value)}
                                className="font-mono text-sm bg-slate-50"
                                placeholder="JANE"
                            />
                        </div>
                        <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold" required>Last Name</Label>
                             <Input 
                                value={formData.lastName || ''} 
                                onChange={(e) => handleChange('lastName', e.target.value)}
                                className="font-mono text-sm bg-slate-50"
                                placeholder="SCHOLAR"
                            />
                        </div>
                         <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold" required>Date of Birth</Label>
                            <DateSelect 
                                value={formData.dob} 
                                onChange={(date) => handleChange('dob', date)}
                                startYear={1950}
                                endYear={new Date().getFullYear() - 15}
                            />
                        </div>
                        <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold">Student ID</Label>
                            <Input 
                                value={formData.studentId || ''} 
                                onChange={(e) => handleChange('studentId', e.target.value)}
                                className="font-mono text-sm"
                                placeholder="12345678"
                            />
                        </div>
                        <div className="space-y-1.5 sm:col-span-2">
                             <Label className="text-xs uppercase text-slate-500 font-semibold flex items-center">
                                <Camera className="w-3 h-3 mr-1" /> Student Photo
                             </Label>
                             <div className="flex items-center space-x-3">
                                <Input 
                                    type="file" 
                                    accept="image/*"
                                    onChange={(e) => handleImageUpload(e, 'studentPhoto')}
                                    className="text-xs file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-xs file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100"
                                />
                                {formData.studentPhoto && <span className="text-xs text-green-600 font-bold flex items-center"><CheckCircle className="w-3 h-3 mr-1"/> Uploaded</span>}
                             </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Academic Enrollment Card */}
                 <Card className="border-t-4 border-t-amber-500 shadow-sm">
                    <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                        <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                            <BookOpen className="w-4 h-4 mr-2" /> 
                            Academic Enrollment
                        </CardTitle>
                    </CardHeader>
                    <CardContent className="pt-6 space-y-4">
                        
                         <TemplateManager 
                            type="INSTITUTION" 
                            label="Institution"
                            onApply={handleApplyInstitutionTemplate}
                            currentData={getCurrentInstitutionState()}
                        />

                        <div className="space-y-1.5 p-3 bg-slate-50 border border-dashed border-slate-300 rounded">
                             <Label className="text-xs uppercase text-slate-500 font-semibold flex items-center">
                                <Shield className="w-3 h-3 mr-1" /> Institution Seal (Optional)
                             </Label>
                             <div className="flex items-center space-x-3">
                                <Input 
                                    type="file" 
                                    accept="image/*"
                                    onChange={(e) => handleImageUpload(e, 'institutionSeal')}
                                    className="text-xs file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-xs file:font-semibold file:bg-amber-50 file:text-amber-700 hover:file:bg-amber-100"
                                />
                                {formData.institutionSeal && <span className="text-xs text-green-600 font-bold flex items-center"><CheckCircle className="w-3 h-3 mr-1"/> Uploaded</span>}
                             </div>
                        </div>

                         <div className="grid gap-4 sm:grid-cols-1">
                            <div className="space-y-1.5">
                                <Label className="text-xs uppercase text-slate-500 font-semibold" required>Institution Name</Label>
                                 <div className="relative">
                                    <Input 
                                        list="institutions"
                                        value={formData.institutionName || ''} 
                                        onChange={(e) => handleChange('institutionName', e.target.value)}
                                        placeholder="SELECT OR TYPE INSTITUTION"
                                    />
                                    <datalist id="institutions">
                                        {DEFAULT_INSTITUTIONS.map(opt => <option key={opt} value={opt} />)}
                                    </datalist>
                                </div>
                            </div>
                        </div>
                         <div className="grid gap-4 sm:grid-cols-2">
                             <div className="space-y-1.5">
                                <Label className="text-xs uppercase text-slate-500 font-semibold" required>Program Name</Label>
                                <Input 
                                    value={formData.programName || ''} 
                                    onChange={(e) => handleChange('programName', e.target.value)}
                                    placeholder="e.g. Computer Science"
                                />
                            </div>
                            <div className="space-y-1.5">
                                <Label className="text-xs uppercase text-slate-500 font-semibold" required>Credential Type</Label>
                                 <Select 
                                    value={formData.credentialType || ''} 
                                    onChange={(e) => handleChange('credentialType', e.target.value)}
                                >
                                    <option value="" disabled>SELECT TYPE</option>
                                    {CREDENTIAL_TYPES.map(opt => <option key={opt} value={opt}>{opt}</option>)}
                                </Select>
                            </div>
                         </div>

                         <div className="grid gap-4 sm:grid-cols-3">
                             <div className="space-y-1.5">
                                 <Label className="text-xs uppercase text-slate-500 font-semibold" required>Term</Label>
                                 <Input 
                                    value={formData.termName || ''} 
                                    onChange={(e) => handleChange('termName', e.target.value)}
                                    placeholder="e.g. Spring 2026"
                                 />
                             </div>
                             <div className="space-y-1.5">
                                 <Label className="text-xs uppercase text-slate-500 font-semibold" required>Start Date</Label>
                                 <DateSelect 
                                    value={formData.termStartDate} 
                                    onChange={(date) => handleChange('termStartDate', date)}
                                    startYear={2020}
                                    endYear={2030}
                                 />
                             </div>
                             <div className="space-y-1.5">
                                 <Label className="text-xs uppercase text-slate-500 font-semibold" required>End Date</Label>
                                 <DateSelect 
                                    value={formData.termEndDate} 
                                    onChange={(date) => handleChange('termEndDate', date)}
                                    startYear={2020}
                                    endYear={2030}
                                 />
                             </div>
                         </div>
                    </CardContent>
                 </Card>
                 
                 {/* Packet Builder Toggles */}
                 <Card className="border-t-4 border-t-purple-500 shadow-sm">
                     <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                        <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                            <FileText className="w-4 h-4 mr-2" /> 
                            Packet Contents
                        </CardTitle>
                    </CardHeader>
                    <CardContent className="pt-6">
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                            {[
                                ['includeIdCard', 'Student ID Card'],
                                ['includeEnrollmentLetter', 'Enrollment Verification Letter'],
                                ['includeClassSchedule', 'Class Schedule'],
                                ['includeTuitionReceipt', 'Tuition Receipt'],
                                ['includeTranscript', 'Academic Transcript'],
                                ['includeRegistrationReceipt', 'Registration Receipt'],
                                ['includeGoodStandingLetter', 'Letter of Good Standing'],
                                ['includeFinancialStatement', 'Financial Statement']
                            ].map(([key, label]) => (
                                <label key={key} className="flex items-center space-x-2 cursor-pointer p-2 rounded hover:bg-slate-50 border border-transparent hover:border-slate-100">
                                    <input 
                                        type="checkbox" 
                                        checked={!!formData[key as keyof StudentVerificationData]} 
                                        onChange={(e) => handleChange(key as keyof StudentVerificationData, e.target.checked)}
                                        className="w-4 h-4 text-indigo-600 rounded border-gray-300 focus:ring-indigo-500"
                                    />
                                    <span className="text-sm text-slate-700">{label}</span>
                                </label>
                            ))}
                        </div>
                    </CardContent>
                 </Card>

                {/* Detailed Data Editors */}
                 <Card className="border-t-4 border-t-cyan-500 shadow-sm">
                     <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                        <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center justify-between">
                            <div className="flex items-center">
                                <List className="w-4 h-4 mr-2" /> 
                                Data Detail Editors
                            </div>
                        </CardTitle>
                    </CardHeader>
                    <CardContent className="pt-0 px-0">
                        {/* Tabs */}
                        <div className="flex border-b border-slate-200 overflow-x-auto" role="tablist" aria-label="Data Editor Tabs">
                            <button 
                                role="tab"
                                aria-selected={activeTab === 'courses'}
                                aria-controls="courses-panel"
                                onClick={() => setActiveTab('courses')}
                                className={`px-4 py-3 text-xs font-bold uppercase tracking-wider border-b-2 transition-colors flex items-center whitespace-nowrap ${activeTab === 'courses' ? 'border-cyan-500 text-cyan-700 bg-cyan-50' : 'border-transparent text-slate-500 hover:text-slate-700 hover:bg-slate-50'}`}
                            >
                                <BookOpen className="w-3 h-3 mr-2" /> Courses
                            </button>
                            <button 
                                role="tab"
                                aria-selected={activeTab === 'transcript'}
                                aria-controls="transcript-panel"
                                onClick={() => setActiveTab('transcript')}
                                className={`px-4 py-3 text-xs font-bold uppercase tracking-wider border-b-2 transition-colors flex items-center whitespace-nowrap ${activeTab === 'transcript' ? 'border-cyan-500 text-cyan-700 bg-cyan-50' : 'border-transparent text-slate-500 hover:text-slate-700 hover:bg-slate-50'}`}
                            >
                                <GraduationCap className="w-3 h-3 mr-2" /> Transcript
                            </button>
                            <button 
                                role="tab"
                                aria-selected={activeTab === 'tuition'}
                                aria-controls="tuition-panel"
                                onClick={() => setActiveTab('tuition')}
                                className={`px-4 py-3 text-xs font-bold uppercase tracking-wider border-b-2 transition-colors flex items-center whitespace-nowrap ${activeTab === 'tuition' ? 'border-cyan-500 text-cyan-700 bg-cyan-50' : 'border-transparent text-slate-500 hover:text-slate-700 hover:bg-slate-50'}`}
                            >
                                <CreditCard className="w-3 h-3 mr-2" /> Tuition
                            </button>
                             <button 
                                role="tab"
                                aria-selected={activeTab === 'registration'}
                                aria-controls="registration-panel"
                                onClick={() => setActiveTab('registration')}
                                className={`px-4 py-3 text-xs font-bold uppercase tracking-wider border-b-2 transition-colors flex items-center whitespace-nowrap ${activeTab === 'registration' ? 'border-cyan-500 text-cyan-700 bg-cyan-50' : 'border-transparent text-slate-500 hover:text-slate-700 hover:bg-slate-50'}`}
                            >
                                <FileText className="w-3 h-3 mr-2" /> Registration
                            </button>
                        </div>

                        {/* Editor Content */}
                        <div className="p-4 bg-slate-50/30 min-h-[300px]">
                            
                            {/* COURSES TAB */}
                            {activeTab === 'courses' && (
                                <div className="space-y-3">
                                    <div className="flex justify-end mb-2">
                                        <Button size="sm" variant="outline" onClick={() => setIsBulkImportOpen(true)} className="text-xs h-7">
                                            <Upload className="w-3 h-3 mr-1" /> Bulk Import
                                        </Button>
                                    </div>
                                    <div className="grid grid-cols-12 gap-2 text-xs font-bold text-slate-500 uppercase px-2 mb-2">
                                        <div className="col-span-2">Code</div>
                                        <div className="col-span-4">Title</div>
                                        <div className="col-span-3">Instructor</div>
                                        <div className="col-span-2">Credits</div>
                                        <div className="col-span-1"></div>
                                    </div>
                                    {formData.courses?.map((course) => (
                                        <div key={course.id} className="grid grid-cols-12 gap-2 items-center bg-white p-2 rounded border border-slate-200 shadow-sm">
                                            <div className="col-span-2">
                                                <Input className="h-8 text-xs font-mono" value={course.code} onChange={(e) => handleUpdateItem('courses', course.id, 'code', e.target.value)} placeholder="CODE" />
                                            </div>
                                            <div className="col-span-4">
                                                <Input className="h-8 text-xs" value={course.title} onChange={(e) => handleUpdateItem('courses', course.id, 'title', e.target.value)} placeholder="Title" />
                                            </div>
                                            <div className="col-span-3">
                                                 <Input className="h-8 text-xs" value={course.instructor} onChange={(e) => handleUpdateItem('courses', course.id, 'instructor', e.target.value)} placeholder="Instructor" />
                                            </div>
                                            <div className="col-span-2">
                                                 <Input type="number" className="h-8 text-xs font-mono" value={course.credits} onChange={(e) => handleUpdateItem('courses', course.id, 'credits', Number.parseFloat(e.target.value))} />
                                            </div>
                                            <div className="col-span-1 flex justify-end">
                                                <Button variant="ghost" size="sm" onClick={() => handleRemoveItem('courses', course.id)} className="h-8 w-8 text-red-400 hover:text-red-600 hover:bg-red-50 p-0">
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                             {/* Extended details for course (2nd row) */}
                                             <div className="col-span-12 grid grid-cols-2 gap-2 mt-1 pt-1 border-t border-slate-50">
                                                 <Input className="h-7 text-[10px]" value={course.days} onChange={(e) => handleUpdateItem('courses', course.id, 'days', e.target.value)} placeholder="Days (e.g. MWF)" />
                                                 <Input className="h-7 text-[10px]" value={course.time} onChange={(e) => handleUpdateItem('courses', course.id, 'time', e.target.value)} placeholder="Time (e.g. 10:00 - 11:30)" />
                                             </div>
                                        </div>
                                    ))}
                                    <Button variant="outline" size="sm" onClick={() => handleAddItem('courses')} className="w-full border-dashed text-slate-500 hover:text-indigo-600 hover:border-indigo-300">
                                        <Plus className="w-4 h-4 mr-2" /> Add Course
                                    </Button>
                                </div>
                            )}

                            {/* TRANSCRIPT TAB */}
                            {activeTab === 'transcript' && (
                                <div className="space-y-3">
                                     <div className="flex justify-end mb-2">
                                        <Button size="sm" variant="outline" onClick={handleSmartFillTranscript} className="text-xs h-7">
                                            <RotateCcw className="w-3 h-3 mr-1" /> Auto-Fill from Courses
                                        </Button>
                                    </div>
                                     <div className="grid grid-cols-12 gap-2 text-xs font-bold text-slate-500 uppercase px-2 mb-2">
                                        <div className="col-span-3">Term</div>
                                        <div className="col-span-4">Course</div>
                                        <div className="col-span-2">Credits</div>
                                        <div className="col-span-2">Grade</div>
                                        <div className="col-span-1"></div>
                                    </div>
                                    {formData.transcriptEntries?.map((entry) => (
                                        <div key={entry.id} className="grid grid-cols-12 gap-2 items-center bg-white p-2 rounded border border-slate-200 shadow-sm">
                                             <div className="col-span-3">
                                                <Input className="h-8 text-xs" value={entry.term} onChange={(e) => handleUpdateItem('transcript', entry.id, 'term', e.target.value)} placeholder="Term" />
                                            </div>
                                            <div className="col-span-4">
                                                <Input className="h-8 text-xs font-mono" value={entry.course} onChange={(e) => handleUpdateItem('transcript', entry.id, 'course', e.target.value)} placeholder="Course Code" />
                                            </div>
                                             <div className="col-span-2">
                                                <Input type="number" className="h-8 text-xs font-mono" value={entry.credits} onChange={(e) => handleUpdateItem('transcript', entry.id, 'credits', Number.parseFloat(e.target.value))} />
                                            </div>
                                            <div className="col-span-2">
                                                <Input className="h-8 text-xs font-mono" value={entry.grade} onChange={(e) => handleUpdateItem('transcript', entry.id, 'grade', e.target.value)} placeholder="Grade" />
                                            </div>
                                            <div className="col-span-1 flex justify-end">
                                                <Button variant="ghost" size="sm" onClick={() => handleRemoveItem('transcript', entry.id)} className="h-8 w-8 text-red-400 hover:text-red-600 hover:bg-red-50 p-0">
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        </div>
                                    ))}
                                    <Button variant="outline" size="sm" onClick={() => handleAddItem('transcript')} className="w-full border-dashed text-slate-500 hover:text-indigo-600 hover:border-indigo-300">
                                        <Plus className="w-4 h-4 mr-2" /> Add Transcript Entry
                                    </Button>
                                </div>
                            )}

                            {/* TUITION TAB */}
                            {activeTab === 'tuition' && (
                                <div className="space-y-3">
                                     <div className="flex justify-end mb-2">
                                        <Button size="sm" variant="outline" onClick={handleCalculateFees} className="text-xs h-7">
                                            <CreditCard className="w-3 h-3 mr-1" /> Auto-Calculate Fees
                                        </Button>
                                    </div>
                                    <div className="grid grid-cols-12 gap-2 text-xs font-bold text-slate-500 uppercase px-2 mb-2">
                                        <div className="col-span-8">Description</div>
                                        <div className="col-span-3">Amount ($)</div>
                                        <div className="col-span-1"></div>
                                    </div>
                                    {formData.tuitionItems?.map((item) => (
                                        <div key={item.id} className="grid grid-cols-12 gap-2 items-center bg-white p-2 rounded border border-slate-200 shadow-sm">
                                            <div className="col-span-8">
                                                <Input className="h-8 text-xs" value={item.description} onChange={(e) => handleUpdateItem('tuition', item.id, 'description', e.target.value)} placeholder="Fee Description" />
                                            </div>
                                            <div className="col-span-3">
                                                <Input type="number" className="h-8 text-xs font-mono" value={item.amount} onChange={(e) => handleUpdateItem('tuition', item.id, 'amount', Number.parseFloat(e.target.value))} placeholder="0.00" />
                                            </div>
                                             <div className="col-span-1 flex justify-end">
                                                <Button variant="ghost" size="sm" onClick={() => handleRemoveItem('tuition', item.id)} className="h-8 w-8 text-red-400 hover:text-red-600 hover:bg-red-50 p-0">
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        </div>
                                    ))}
                                     <Button variant="outline" size="sm" onClick={() => handleAddItem('tuition')} className="w-full border-dashed text-slate-500 hover:text-indigo-600 hover:border-indigo-300">
                                        <Plus className="w-4 h-4 mr-2" /> Add Item
                                    </Button>
                                </div>
                            )}

                             {/* REGISTRATION TAB */}
                            {activeTab === 'registration' && (
                                <div className="space-y-3">
                                    <div className="grid grid-cols-12 gap-2 text-xs font-bold text-slate-500 uppercase px-2 mb-2">
                                        <div className="col-span-8">Description</div>
                                        <div className="col-span-3">Amount ($)</div>
                                        <div className="col-span-1"></div>
                                    </div>
                                    {formData.registrationItems?.map((item) => (
                                        <div key={item.id} className="grid grid-cols-12 gap-2 items-center bg-white p-2 rounded border border-slate-200 shadow-sm">
                                            <div className="col-span-8">
                                                <Input className="h-8 text-xs" value={item.description} onChange={(e) => handleUpdateItem('registration', item.id, 'description', e.target.value)} placeholder="Fee Description" />
                                            </div>
                                            <div className="col-span-3">
                                                <Input type="number" className="h-8 text-xs font-mono" value={item.amount} onChange={(e) => handleUpdateItem('registration', item.id, 'amount', Number.parseFloat(e.target.value))} placeholder="0.00" />
                                            </div>
                                             <div className="col-span-1 flex justify-end">
                                                <Button variant="ghost" size="sm" onClick={() => handleRemoveItem('registration', item.id)} className="h-8 w-8 text-red-400 hover:text-red-600 hover:bg-red-50 p-0">
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        </div>
                                    ))}
                                     <Button variant="outline" size="sm" onClick={() => handleAddItem('registration')} className="w-full border-dashed text-slate-500 hover:text-indigo-600 hover:border-indigo-300">
                                        <Plus className="w-4 h-4 mr-2" /> Add Item
                                    </Button>
                                </div>
                            )}

                        </div>
                    </CardContent>
                 </Card>
            </div>

            {/* Right Col: Admin & Actions */}
            <div className="space-y-6 flex flex-col">
                 {/* Admin Controls */}
                 <Card className="border-t-4 border-t-slate-500 shadow-sm">
                    <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                        <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                            <Settings className="w-4 h-4 mr-2" /> 
                            Admin Controls
                        </CardTitle>
                    </CardHeader>
                    <CardContent className="pt-6 space-y-4">
                         <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold" required>Contact Email</Label>
                            <Input 
                               value={formData.email || ''} 
                               onChange={(e) => handleChange('email', e.target.value)}
                               className="font-mono text-sm"
                               placeholder="registrar@institution.edu"
                            />
                        </div>
                        
                         {/* PDF SETTINGS SECTION */}
                         <div className="pt-4 border-t border-dashed border-gray-200">
                             <Label className="text-xs uppercase text-slate-500 font-semibold mb-2 block">PDF Settings</Label>
                             <div className="grid grid-cols-2 gap-2">
                                <Select 
                                   value={pdfSettings.paperSize} 
                                   onChange={(e) => setPdfSettings(prev => ({...prev, paperSize: e.target.value as any}))} 
                                >
                                   <option value="LETTER">Letter (8.5x11)</option>
                                   <option value="A4">A4 (210x297)</option>
                                </Select>

                             </div>
                             <div className="mt-2 flex items-center space-x-2">
                                <input 
                                   type="checkbox" 
                                   checked={pdfSettings.includePageNumbers} 
                                   onChange={(e) => setPdfSettings(prev => ({...prev, includePageNumbers: e.target.checked}))}
                                   className="rounded border-gray-300"
                                />
                                <span className="text-xs text-slate-600">Include Page Numbers & Footer</span>
                             </div>
                         </div>
                         
                         {/* ID CARD SETTINGS */}
                         <div className="pt-4 border-t border-dashed border-gray-200">
                             <Label className="text-xs uppercase text-slate-500 font-semibold mb-2 block">ID Card Style</Label>
                             <div className="grid grid-cols-2 gap-2">
                                <Select 
                                   value={idSettings.theme} 
                                   onChange={(e) => setIdSettings(prev => ({...prev, theme: e.target.value as any}))} 
                                >
                                   <option value="LIGHT">Light Theme</option>
                                   <option value="DARK">Dark Theme</option>
                                   <option value="HIGH_CONTRAST">High Contrast</option>
                                </Select>
                                <Select 
                                   value={idSettings.barcodeType} 
                                   onChange={(e) => setIdSettings(prev => ({...prev, barcodeType: e.target.value as any}))} 
                                >
                                   <option value="QR">QR Code</option>
                                   <option value="CODE128">Barcode 128</option>
                                </Select>
                             </div>
                         </div>

                        <div className="pt-4 border-t border-dashed border-gray-200">
                             <Label className="text-xs uppercase text-slate-500 font-semibold mb-2 block">Custom Assets</Label>
                             
                             <div className="grid grid-cols-2 gap-2">
                                 <div className="border border-slate-200 rounded p-2 text-center hover:bg-slate-50 transition-colors cursor-pointer relative overflow-hidden group">
                                     <input type="file" className="absolute inset-0 opacity-0 cursor-pointer" onChange={(e) => handleImageUpload(e, 'institutionSeal')} accept="image/*" />
                                     <Shield className="w-6 h-6 mx-auto text-slate-400 mb-1 group-hover:text-indigo-500" />
                                     <span className="text-[10px] uppercase font-bold text-slate-500">Upload Seal</span>
                                 </div>
                                 <div className="border border-slate-200 rounded p-2 text-center hover:bg-slate-50 transition-colors cursor-pointer relative overflow-hidden group">
                                     <input type="file" className="absolute inset-0 opacity-0 cursor-pointer" onChange={(e) => handleImageUpload(e, 'studentPhoto')} accept="image/*" />
                                     <Camera className="w-6 h-6 mx-auto text-slate-400 mb-1 group-hover:text-amber-500" />
                                      <span className="text-[10px] uppercase font-bold text-slate-500">Student Photo</span>
                                 </div>
                             </div>
                        </div>
                        
                         <div className="pt-4 border-t border-dashed border-gray-200 flex gap-2">
                           <Button variant="outline" size="sm" onClick={handleImportJSON} className="flex-1 text-xs">
                             <Upload size={14} className="mr-2" /> Import
                           </Button>
                           <Button variant="outline" size="sm" onClick={handleExportJSON} className="flex-1 text-xs">
                             <Save size={14} className="mr-2" /> Export
                           </Button>
                         </div>
                    </CardContent>
                 </Card>
    
                  {/* Status & Actions */}
                 <div className="bg-slate-900 rounded-lg p-4 text-white shadow-lg">
                    <div className="mb-4">
                        <h3 className="text-xs font-mono text-indigo-400 mb-1">SYSTEM STATUS</h3>
                        <div className="flex items-center space-x-2">
                            <div className={`w-2 h-2 rounded-full ${canGenerate ? 'bg-emerald-500 animate-pulse' : 'bg-amber-500'}`}></div>
                            <span className="text-sm font-bold tracking-wider">{canGenerate ? 'READY FOR PROCESS' : 'AWAITING DATA'}</span>
                        </div>
                        {!canGenerate && (
                            <div className="flex items-center mt-2 space-x-2">
                                <input type="checkbox" checked={forceGenerate} onChange={(e) => setForceGenerate(e.target.checked)} className="rounded border-slate-600 bg-slate-800" />
                                <span className="text-[10px] text-amber-500">Override Validation (Demo Mode)</span>
                            </div>
                        )}
                    </div>
                    
                    {successMessage && (
                        <div className="mb-4 p-2 bg-emerald-900/50 border border-emerald-500/30 rounded text-xs text-emerald-300 font-mono flex items-center">
                             <CheckCircle className="w-4 h-4 mr-2" />
                             {successMessage}
                        </div>
                    )}
                    {errorMessage && (
                        <div className="mb-4 p-2 bg-red-900/50 border border-red-500/30 rounded text-xs text-red-300 font-mono flex items-center">
                             <AlertCircle className="w-4 h-4 mr-2" />
                             {errorMessage}
                        </div>
                    )}
    
                    <Button 
                        className={`w-full font-bold tracking-wider ${!canGenerate ? 'bg-slate-700 text-slate-400' : 'bg-indigo-600 hover:bg-indigo-500 text-white'}`}
                        size="lg" 
                        onClick={handleDownloadClick} 
                        disabled={isGenerating || (!canGenerate)}
                    >
                        {isGenerating ? (
                            <span className="flex items-center"><Loader2 className="mr-2 h-4 w-4 animate-spin" /> PROCESSING...</span>
                        ) : (
                            <span className="flex items-center justify-center"><Download className="mr-2 h-4 w-4" /> GENERATE PACKET</span>
                        )}
                    </Button>
                 </div>
    
                 <SystemTerminal logs={logs} />
            </div>
          </div>
      )}
    </div>
  );
}
