import React from 'react';
import { Page, Text, View, Document, StyleSheet, Image, Font } from '@react-pdf/renderer';
import { StudentVerificationData, PdfSettings, IdCardSettings } from '@/types/student-verification';

// Helper to simulate a barcode 128 (just visual lines)
// Helper to simulate a barcode 128 (deterministic visual lines)
const BarcodeSimulation = ({ value }: { value: string }) => {
    const getHash = (str: string) => {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return Math.abs(hash);
    };

    const seed = getHash(value || 'DEFAULT');
    const bars = Array.from({ length: 50 }).map((_, i) => {
        const val = (seed + i * 12345) % 100;
        return {
            width: val > 55 ? 2 : 1, // Bar width variation
            margin: val % 3 === 0 ? 1.5 : 0.5 // Spacing variation
        };
    });

    return (
        <View style={{ alignItems: 'center', backgroundColor: '#fff', padding: 4 }}>
            <View style={{ flexDirection: 'row', alignItems: 'stretch', height: 30, overflow: 'hidden' }}>
                {bars.map((bar, i) => (
                    <View 
                        key={i} 
                        style={{ 
                            width: bar.width, 
                            height: '100%', 
                            backgroundColor: '#000', 
                            marginRight: bar.margin 
                        }} 
                    />
                ))}
            </View>
            <Text style={{ fontSize: 9, fontFamily: 'Helvetica', marginTop: 2, letterSpacing: 1 }}>{value}</Text>
        </View>
    );
};

const QrCodeSimulation = ({ value, size = 45 }: { value: string, size?: number }) => {
    // Generate a fixed visual pattern for the QR code based on the value
    const cellSize = size / 21; // Standard Version 1 QR is 21x21 modules
    
    // Deterministic random generator
    const getBit = (x: number, y: number) => {
        // Finder patterns (Top-Left, Top-Right, Bottom-Left)
        if ((x < 7 && y < 7) || (x > 13 && y < 7) || (x < 7 && y > 13)) {
            // Outer box
            if ((x === 0 || x === 6) && y <= 6) return true; 
            if ((y === 0 || y === 6) && x <= 6) return true;
            if ((x === 14 || x === 20) && y <= 6) return true;
            if ((y === 0 || y === 6) && x >= 14) return true;
            if ((x === 0 || x === 6) && y >= 14) return true;
            if ((y === 14 || y === 20) && x <= 6) return true;
            
            // Inner block
            if ((x >= 2 && x <= 4) && (y >= 2 && y <= 4)) return true;
            if ((x >= 16 && x <= 18) && (y >= 2 && y <= 4)) return true;
            if ((x >= 2 && x <= 4) && (y >= 16 && y <= 18)) return true;
            
            return false;
        }
        
        // Random data area
        const hash = (x * 31 + y * 17 + value.length) % 100;
        return hash > 45;
    };

    return (
        <View style={{ width: size, height: size, backgroundColor: '#fff', padding: 2, flexDirection: 'column' }}>
            {Array.from({ length: 21 }).map((_, y) => (
                <View key={y} style={{ flexDirection: 'row', height: cellSize }}>
                    {Array.from({ length: 21 }).map((_, x) => (
                        <View 
                            key={`${y}-${x}`}
                            style={{ 
                                width: cellSize, 
                                height: cellSize, 
                                backgroundColor: getBit(x, y) ? '#000' : '#fff' 
                            }} 
                        />
                    ))}
                </View>
            ))}
        </View>
    );
};

const formatDate = (date: Date | null | undefined | string) => {
    if (!date) return 'N/A';
    try {
        return new Date(date).toLocaleDateString('en-US', { month: '2-digit', day: '2-digit', year: 'numeric' });
    } catch { return 'INVALID DATE'; }
};

const ChipSimulation = () => (
    <View style={{ width: 34, height: 25, backgroundColor: '#fcd34d', borderRadius: 5, flexDirection: 'row', flexWrap: 'wrap', padding: 1.5, justifyContent: 'space-between' }}>
        {/* Top Left Pad */}
        <View style={{ width: 14.5, height: 10.5, backgroundColor: 'transparent', borderRightWidth: 0.5, borderBottomWidth: 0.5, borderColor: '#ca8a04' }} />
        {/* Top Right Pad */}
        <View style={{ width: 14.5, height: 10.5, backgroundColor: 'transparent', borderLeftWidth: 0.5, borderBottomWidth: 0.5, borderColor: '#ca8a04' }} />
        {/* Bottom Left Pad */}
        <View style={{ width: 14.5, height: 10.5, backgroundColor: 'transparent', borderRightWidth: 0.5, borderTopWidth: 0.5, borderColor: '#ca8a04' }} />
        {/* Bottom Right Pad */}
        <View style={{ width: 14.5, height: 10.5, backgroundColor: 'transparent', borderLeftWidth: 0.5, borderTopWidth: 0.5, borderColor: '#ca8a04' }} />
        
        {/* Center Cross Detail */}
        <View style={{ position: 'absolute', top: 12, left: 2, right: 2, height: 1, backgroundColor: '#b45309', opacity: 0.3 }} />
        <View style={{ position: 'absolute', left: 16.5, top: 2, bottom: 2, width: 1, backgroundColor: '#b45309', opacity: 0.3 }} />
    </View>
);

interface PdfProps {
  data: Partial<StudentVerificationData>;
  settings: PdfSettings;
  idSettings: IdCardSettings;
  systemRef: string;
}

export const StudentPacketPdf: React.FC<PdfProps> = ({ data, settings, idSettings, systemRef }) => {
    
    // Dynamic Styles based on Settings
    const styles = StyleSheet.create({
      page: {
        flexDirection: 'column',
        backgroundColor: '#FFFFFF',
        padding: settings.margins === 'COMPACT' ? 25 : 50,
        fontFamily: 'Helvetica',
        fontSize: 10,
        position: 'relative',
      },

      footer: {
          display: 'none',
      },
      
      // Headers
      header: {
        marginBottom: 20,
        borderBottomWidth: 2,
        borderBottomColor: '#333',
        paddingBottom: 10,
        flexDirection: 'row', // Ensure items are row-aligned
        justifyContent: 'space-between',
        alignItems: 'center',
      },
      headerTitle: {
          fontSize: 18,
          fontFamily: 'Helvetica-Bold',
          textTransform: 'uppercase',
      },
      headerSub: {
          fontSize: 10,
          fontFamily: 'Helvetica',
          color: '#666',
      },
      sectionTitle: {
          fontFamily: 'Helvetica-Bold', 
          fontSize: 12, 
          marginBottom: 10, 
          marginTop: 20,
          textTransform: 'uppercase',
          color: '#333',
          borderBottomWidth: 1,
          borderBottomColor: '#eee',
          paddingBottom: 2
      },

      // Tables
      table: {
        marginTop: 10,
        marginBottom: 10,
        width: '100%',
        borderWidth: 1,
        borderColor: '#eee',
      },
      tableRow: {
        flexDirection: 'row',
        borderBottomWidth: 1,
        borderBottomColor: '#eee',
        minHeight: 22,
        alignItems: 'center',
      },
      tableHeader: {
        backgroundColor: '#f8f9fa',
        fontFamily: 'Helvetica-Bold',
      },
      tableCell: {
        padding: 5,
        fontSize: 9,
        flex: 1,
      },
      
      // ID Card - Themed
      idCard: {
          width: 336, // CR80 standard ratio approx
          height: 212,
          borderWidth: 1,
          borderColor: idSettings.theme === 'DARK' ? '#333' : '#ccc',
          borderRadius: 10,
          padding: 15,
          backgroundColor: idSettings.theme === 'DARK' ? '#1e293b' : idSettings.theme === 'HIGH_CONTRAST' ? '#000' : '#f8fafc',
          color: idSettings.theme === 'DARK' || idSettings.theme === 'HIGH_CONTRAST' ? '#fff' : '#000',
          marginVertical: 20,
          alignSelf: 'center',
          position: 'relative',
          overflow: 'hidden',
      },
      idCardHeader: {
          flexDirection: 'row',
          justifyContent: 'space-between',
          marginBottom: 10,
          zIndex: 2,
      },
      idCardPhoto: {
          width: 80,
          height: 100,
          backgroundColor: '#ddd',
          marginRight: 15,
          borderWidth: 1,
          borderColor: '#eee',
      },
      
      // Letter
      letterBody: {
          fontSize: 11,
          fontFamily: 'Times-Roman',
          lineHeight: 1.5,
          marginTop: 15,
          marginBottom: 5,
          textAlign: 'justify',
      },
      signatureBlock: { marginTop: 40 },
    });



    // Crypto-style hash for audit
    const getAuditHash = (page: number) => {
        const str = `${systemRef}-${data.studentId}-${page}-${new Date().toISOString().split('T')[0]}`;
        let hash = 0, i, chr;
        if (str.length === 0) return hash;
        for (i = 0; i < str.length; i++) {
            chr = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + chr;
            hash |= 0; 
        }
        return Math.abs(hash).toString(16).toUpperCase().padStart(8, '0');
    };

    const PageFooter = ({ pageNumber }: { pageNumber?: number }) => (
        null
    );

    const SectionDivider = ({ title, subtitle }: { title: string, subtitle?: string }) => (
        <Page size={settings.paperSize} style={[styles.page, { justifyContent: 'center', alignItems: 'center', backgroundColor: '#f8fafc' }]}>

             <View style={{ alignItems: 'center', padding: 40, borderWidth: 4, borderColor: '#333', borderStyle: 'dotted' }}>
                <Text style={{ fontSize: 24, fontFamily: 'Helvetica-Bold', textTransform: 'uppercase', marginBottom: 10 }}>{title}</Text>
                {subtitle && <Text style={{ fontSize: 14, color: '#666', fontFamily: 'Helvetica' }}>{subtitle}</Text>}
             </View>
             <PageFooter />
        </Page>
    );
    


    return (
      <Document>
        {/* PAGE 1: COVER SUMMARY */}
        <Page size={settings.paperSize} style={styles.page}>

            <View style={styles.header}>
                <View>
                    <Text style={styles.headerTitle}>Official Academic Record</Text>
                    <Text style={styles.headerSub}>Generated: {new Date().toLocaleString()}</Text>
                </View>
                {data.institutionSeal && (
                    <Image src={data.institutionSeal} style={{ width: 50, height: 50, objectFit: 'contain' }} />
                )}
            </View>

            {/* TABLE OF CONTENTS */}
            {settings.includeTableOfContents && (
                <View style={{ marginTop: 30, marginBottom: 30, padding: 20, backgroundColor: '#f8fafc', borderLeftWidth: 4, borderLeftColor: '#334155' }}>
                     <Text style={[styles.sectionTitle, { marginTop: 0, borderBottomWidth: 0 }]}>Packet Contents</Text>
                     <View style={{ marginLeft: 10 }}>
                         <Text style={{ marginBottom: 5 }}>• Student Summary Profile</Text>
                         {data.includeIdCard && <Text style={{ marginBottom: 5 }}>• Student Identification Card</Text>}
                         {data.includeEnrollmentLetter && <Text style={{ marginBottom: 5 }}>• Official Enrollment Verification Letter</Text>}
                         {data.includeClassSchedule && <Text style={{ marginBottom: 5 }}>• Academic Course Schedule</Text>}
                         {data.includeTuitionReceipt && <Text style={{ marginBottom: 5 }}>• Tuition & Fee Receipt</Text>}
                         {data.includeTranscript && <Text style={{ marginBottom: 5 }}>• Unofficial Academic Transcript</Text>}
                         {data.includeRegistrationReceipt && <Text style={{ marginBottom: 5 }}>• Registration Acknowledgment</Text>}
                         {data.includeGoodStandingLetter && <Text style={{ marginBottom: 5 }}>• Certificate of Good Standing</Text>}
                         <Text style={{ marginTop: 10, fontSize: 8, color: '#666', fontStyle: 'italic' }}>* This packet contains verified data generated directly from the {data.institutionName} Student Information System.</Text>
                     </View>
                </View>
            )}

            <View style={{ marginTop: 10 }}>
                <Text style={styles.sectionTitle}>STUDENT SUMMARY</Text>
                <View style={styles.table}>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Full Name</Text>
                        <Text style={[styles.tableCell, { flex: 2, fontFamily: 'Helvetica-Bold' }]}>{data.lastName}, {data.firstName}</Text>
                    </View>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Student ID</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{data.studentId}</Text>
                    </View>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Date of Birth</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{formatDate(data.dob)}</Text>
                    </View>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Email</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{data.email}</Text>
                    </View>
                </View>
                
                <Text style={styles.sectionTitle}>ENROLLMENT SUMMARY</Text>
                 <View style={styles.table}>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Institution</Text>
                        <Text style={[styles.tableCell, { flex: 2, fontFamily: 'Helvetica-Bold' }]}>{data.institutionName}</Text>
                    </View>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Program</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{data.programName}</Text>
                    </View>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Credential</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{data.credentialType}</Text>
                    </View>
                     <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Status</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{data.enrollmentStatus}</Text>
                    </View>
                     <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Current Term</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{data.termName}</Text>
                    </View>
                    <View style={styles.tableRow}>
                        <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Term Dates</Text>
                        <Text style={[styles.tableCell, { flex: 2 }]}>{formatDate(data.termStartDate)} - {formatDate(data.termEndDate)}</Text>
                    </View>
                </View>
                
                <View style={{ marginTop: 20, padding: 10, backgroundColor: '#f0fdf4', borderWidth: 1, borderColor: '#16a34a' }}>
                     <Text style={{ color: '#166534', fontFamily: 'Helvetica-Bold', marginBottom: 5 }}>VERIFICATION STATUS: CONFIRMED</Text>
                     <Text style={{ color: '#166534', fontSize: 9 }}>
                         The student is currently enrolled in credit-bearing courses toward a diploma, degree, or professional license.
                     </Text>
                </View>
            </View>
            <Text 
                render={({ pageNumber, totalPages }) => (
                    <Text style={{ position: 'absolute', bottom: 30, right: 30, fontSize: 8, color: '#999' }}>
                         {pageNumber} / {totalPages}
                    </Text>
                )} 
                fixed 
            />
            <PageFooter pageNumber={1} />
        </Page>

        {/* PAGE 2: STUDENT ID CARD */}
        {data.includeIdCard && (
            <Page size={settings.paperSize} style={styles.page}>

                <Text style={styles.headerTitle}>Identification Card</Text>
                
                <View style={styles.idCard}>
                    {/* Background decoration */}
                    <View style={{ position: 'absolute', top: 0, left: 0, right: 0, height: 15, backgroundColor: idSettings.theme === 'DARK' ? '#334155' : '#1e40af' }} />
                    <View style={{ position: 'absolute', top: -50, right: -50, width: 200, height: 200, borderRadius: 100, backgroundColor: idSettings.theme === 'DARK' ? '#334155' : '#e2e8f0', opacity: 0.5 }} />
                     


                      <View style={[styles.idCardHeader, { marginTop: 20 }]}>
                         <View>
                            <Text style={{ fontSize: 13, fontFamily: 'Helvetica-Bold', color: idSettings.theme === 'HIGH_CONTRAST' ? '#fff' : '#1e40af', width: 220 }}>{data.institutionName}</Text>
                            <Text style={{ fontSize: 7, marginTop: 2, color: idSettings.theme === 'DARK' ? '#94a3b8' : '#64748b' }}>STUDENT IDENTITY</Text>
                         </View>
                         {data.institutionSeal && (
                            <Image src={data.institutionSeal} style={{ width: 35, height: 35, objectFit: 'contain' }} />
                         )}
                     </View>
                     
                     <View style={{ flexDirection: 'row', zIndex: 2 }}>
                         <View style={{ position: 'relative' }}>
                             {data.studentPhoto ? (
                                 <Image src={data.studentPhoto} style={styles.idCardPhoto} />
                             ) : (
                                 <View style={[styles.idCardPhoto, { alignItems: 'center', justifyContent: 'center' }]}>
                                     <Text style={{ fontSize: 8, color: '#999' }}>PHOTO</Text>
                                 </View>
                             )}
                             <View style={{ position: 'absolute', bottom: 6, left: 6 }}>
                                 <ChipSimulation />
                             </View>
                         </View>
                         <View style={{ marginLeft: 10, justifyContent: 'center', flex: 1 }}>
                             <Text style={{ fontFamily: 'Helvetica-Bold', fontSize: 14, marginBottom: 5 }}>{data.firstName} {data.lastName}</Text>
                             
                             <Text style={{ fontSize: 9, marginBottom: 1 }}>ID: {data.studentId}</Text>
                             <Text style={{ fontSize: 9, marginBottom: 1 }}>Program: {data.programName}</Text>
                             <Text style={{ fontSize: 9, marginBottom: 5 }}>Level: {data.credentialType} – {data.termName}</Text>

                             <Text style={{ fontSize: 8, color: idSettings.theme === 'DARK' ? '#cbd5e1' : '#475569' }}>
                                 Issued: {data.termStartDate ? formatDate(data.termStartDate) : formatDate(new Date())}
                             </Text>
                             <Text style={{ fontSize: 9, marginTop: 2, fontFamily: 'Helvetica-Bold', color: idSettings.theme === 'DARK' ? '#f472b6' : '#c026d3' }}>
                                 Expires: {data.expectedGraduationDate ? formatDate(data.expectedGraduationDate) : formatDate(new Date(new Date().setFullYear(new Date().getFullYear() + 1)))}
                             </Text>
                         </View>
                     </View>
                     <View style={{ position: 'absolute', bottom: 12, right: 15, alignItems: 'flex-end', zIndex: 2 }}>
                         {idSettings.barcodeType === 'CODE128' ? (
                            <BarcodeSimulation value={data.studentId || 'UNKNOWN'} />
                         ) : (
                            <QrCodeSimulation value={data.studentId || 'QR'} />
                         )}
                     </View>
                </View>
                <PageFooter />
            </Page>
        )}

        {/* ACADEMIC SECTION */}
        
        {/* PAGE 3: ENROLLMENT VERIFICATION LETTER */}
        {data.includeEnrollmentLetter && (
            <Page size={settings.paperSize} style={styles.page}>

                 <View style={{ borderBottomWidth: 1, borderBottomColor: '#000', paddingBottom: 15, marginBottom: 20, alignItems: 'center' }}>
                    {data.institutionSeal && (
                         <Image src={data.institutionSeal} style={{ width: 60, height: 60, marginBottom: 10, objectFit: 'contain' }} />
                    )}
                     <Text style={{ fontSize: 18, fontFamily: 'Times-Bold', textTransform: 'uppercase' }}>{data.institutionName}</Text>
                     <Text style={{ fontSize: 10, marginTop: 5 }}>Office of the Registrar</Text>
                </View>
                
                <Text style={{ fontSize: 11, fontFamily: 'Times-Roman', alignSelf: 'flex-end' }}>Date: {new Date().toLocaleDateString()}</Text>
                
                <Text style={{ fontSize: 11, fontFamily: 'Times-Bold', marginTop: 20, marginBottom: 10 }}>TO WHOM IT MAY CONCERN:</Text>
                
                <Text style={styles.letterBody}>
                    This letter confirms that <Text style={{ fontFamily: 'Times-Bold' }}>{data.firstName} {data.lastName}</Text> (Student ID: {data.studentId}) is currently enrolled as a student at {data.institutionName}.
                </Text>
                
                <Text style={styles.letterBody}>
                    The student is currently active for the <Text style={{ fontFamily: 'Times-Bold' }}>{data.termName}</Text> term, which began on {formatDate(data.termStartDate)} and ends on {formatDate(data.termEndDate)}. 
                    They are pursuing a {data.credentialType} in {data.programName}.
                </Text>

                <Text style={styles.letterBody}>
                    The student is enrolled in credit-bearing coursework totaling {data.creditsThisTerm} credit hours for this term, which counts toward their degree or professional credential.
                </Text>

                 <View style={styles.signatureBlock}>
                      <Text style={{ fontSize: 11, fontFamily: 'Times-Roman' }}>Sincerely,</Text>
                      {/* Fake Signature */}
                      <Text style={{ fontFamily: 'Times-Roman', fontSize: 18, marginTop: 15, marginBottom: 5, color: '#000080', fontStyle: 'italic' }}>
                          {data.registrarContact?.split('|')[0] || 'University Registrar'}
                      </Text> 
                      <Text style={{ fontSize: 11, fontFamily: 'Times-Bold' }}>Office of the Registrar</Text>
                      <Text style={{ fontSize: 11, fontFamily: 'Times-Roman' }}>{data.institutionName}</Text>
                      <Text style={{ fontSize: 10, fontFamily: 'Times-Roman', color: '#666', marginTop: 5 }}>{data.registrarContact || 'registrar@institution.edu'}</Text>
                 </View>

                <PageFooter />
            </Page>
        )}

        {/* PAGE 4: CLASS SCHEDULE */}
        {data.includeClassSchedule && (
            <Page size={settings.paperSize} style={styles.page}>

                <View style={{ alignItems: 'center', marginBottom: 20 }}>
                     <Text style={{ fontSize: 9, color: '#555', marginBottom: 5, textTransform: 'uppercase', letterSpacing: 1 }}>Office of the Registrar • Student Record</Text>
                     <Text style={{ fontSize: 18, fontFamily: 'Helvetica-Bold', textAlign: 'center', color: '#000' }}>
                        OFFICIAL CLASS SCHEDULE
                     </Text>
                     <Text style={{ fontSize: 12, color: '#333', marginTop: 2 }}>{data.termName || 'Spring 2026'}</Text>
                </View>

                {/* Student Data Box */}
                <View style={{ borderWidth: 1, borderColor: '#333', padding: 15, marginBottom: 20, backgroundColor: '#f9f9f9' }}>
                    <Text style={{ color: '#000', fontFamily: 'Helvetica-Bold', fontSize: 11, marginBottom: 8, textTransform: 'uppercase', borderBottomWidth: 1, borderBottomColor: '#ddd', paddingBottom: 4 }}>Student Information</Text>
                    
                    <View style={{ flexDirection: 'row', flexWrap: 'wrap' }}>
                        <View style={{ width: '50%', marginBottom: 6 }}>
                             <Text style={{ fontSize: 9, color: '#555' }}>Student Name</Text>
                             <Text style={{ fontSize: 10, fontFamily: 'Helvetica-Bold' }}>{(data.firstName || '').toUpperCase()} {(data.lastName || '').toUpperCase()}</Text>
                        </View>
                        <View style={{ width: '50%', marginBottom: 6 }}>
                             <Text style={{ fontSize: 9, color: '#555' }}>Student ID</Text>
                             <Text style={{ fontSize: 10, fontFamily: 'Helvetica-Bold' }}>{data.studentId}</Text>
                        </View>
                        <View style={{ width: '50%', marginBottom: 6 }}>
                             <Text style={{ fontSize: 9, color: '#555' }}>Program of Study</Text>
                             <Text style={{ fontSize: 10 }}>{data.programName}</Text>
                        </View>
                         <View style={{ width: '50%', marginBottom: 6 }}>
                             <Text style={{ fontSize: 9, color: '#555' }}>Credential Type</Text>
                             <Text style={{ fontSize: 10 }}>{data.credentialType || 'Degree Seeking'}</Text>
                        </View>
                    </View>
                </View>
                
                {/* Schedule Table */}
                <View style={[styles.table, { borderTopWidth: 1, borderTopColor: '#000', borderLeftWidth: 0, borderRightWidth: 0 }]}>
                    <View style={[styles.tableRow, { borderBottomWidth: 1, borderBottomColor: '#000', backgroundColor: '#f0f0f0', paddingVertical: 6 }]}>
                        <Text style={[styles.tableCell, { flex: 4, color: '#000', fontFamily: 'Helvetica-Bold', fontSize: 10, textTransform: 'uppercase' }]}>Course Details</Text>
                        <Text style={[styles.tableCell, { flex: 2, color: '#000', fontFamily: 'Helvetica-Bold', fontSize: 10, textTransform: 'uppercase' }]}>Meeting Days</Text>
                        <Text style={[styles.tableCell, { flex: 2, color: '#000', fontFamily: 'Helvetica-Bold', fontSize: 10, textTransform: 'uppercase' }]}>Time</Text>
                        <Text style={[styles.tableCell, { flex: 2, color: '#000', fontFamily: 'Helvetica-Bold', fontSize: 10, textTransform: 'uppercase', textAlign: 'right' }]}>Location</Text>
                    </View>
                    {data.courses?.map((course, index) => (
                        <View key={course.id} style={[styles.tableRow, { borderBottomWidth: 1, borderBottomColor: '#eee', paddingVertical: 10 }]}>
                            <Text style={[styles.tableCell, { flex: 4, fontFamily: 'Helvetica' }]}>
                                <Text style={{ fontFamily: 'Helvetica-Bold' }}>{course.code}</Text> – {course.title}
                            </Text>
                            <Text style={[styles.tableCell, { flex: 2, fontSize: 10 }]}>{course.days}</Text>
                            <Text style={[styles.tableCell, { flex: 2, fontSize: 10 }]}>{course.time}</Text>
                            <Text style={[styles.tableCell, { flex: 2, fontSize: 10, textAlign: 'right' }]}>
                                {index % 2 === 0 ? 'Room A-204' : 'Lab E-05'}
                            </Text>
                        </View>
                    ))}
                </View>

                {/* Footer Notes */}
                <View style={{ marginTop: 25, borderTopWidth: 2, borderTopColor: '#000', paddingTop: 10 }}>
                     <View style={{ flexDirection: 'row', justifyContent: 'space-between', marginBottom: 5 }}>
                        <Text style={{ fontSize: 9, fontFamily: 'Helvetica-Bold' }}>Term Status: <Text style={{ fontFamily: 'Helvetica', color: '#166534' }}>ACTIVE</Text></Text>
                        <Text style={{ fontSize: 9, color: '#555' }}>Date Issued: {new Date().toLocaleDateString('en-US')}</Text>
                     </View>
                     
                     <Text style={{ fontSize: 9, color: '#444', fontStyle: 'italic', lineHeight: 1.4 }}>
                         * This document verifies the student's registration for the term indicated. Course times and locations are subject to change. 
                         Official grades are released at the end of the term.
                     </Text>
                     
                     <Text style={{ fontSize: 9, color: '#000', marginTop: 10 }}>
                         <Text style={{ fontFamily: 'Helvetica-Bold' }}>Term Dates:</Text> {data.termStartDate ? new Date(data.termStartDate).toLocaleDateString('en-US') : '08/25/2025'} — {data.termEndDate ? new Date(data.termEndDate).toLocaleDateString('en-US') : '12/12/2025'}
                     </Text>
                </View>

                <View style={{ position: 'absolute', bottom: 40, left: 50, right: 50, alignItems: 'center' }}>
                    <Text style={{ fontSize: 8, color: '#888', textAlign: 'center' }}>
                        {(data.institutionName || 'UNIVERSITY').toUpperCase()} • OFFICE OF THE REGISTRAR • OFFICIAL RECORD
                    </Text>
                </View>

                <PageFooter />
            </Page>
        )}

        {/* FINANCIAL SECTION */}
        
        {/* PAGE 5: TUITION RECEIPT */}
        {data.includeTuitionReceipt && (
            <Page size={settings.paperSize} style={styles.page}>

                <Text style={styles.headerTitle}>Tuition & Fee Receipt</Text>
                <Text style={styles.headerSub}>Receipt #: {Math.floor(Math.random() * 1000000)}</Text>
                 <Text style={styles.headerSub}>Date: {formatDate(data.termStartDate)}</Text>
                
                 <View style={[styles.table, { marginTop: 30 }]}>
                    <View style={[styles.tableRow, styles.tableHeader]}>
                        <Text style={[styles.tableCell, { flex: 3 }]}>Description</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'right' }]}>Amount (USD)</Text>
                    </View>
                     {data.tuitionItems?.map((item) => (
                        <View key={item.id} style={styles.tableRow}>
                            <Text style={[styles.tableCell, { flex: 3 }]}>{item.description}</Text>
                            <Text style={[styles.tableCell, { flex: 1, textAlign: 'right' }]}>{item.amount.toFixed(2)}</Text>
                        </View>
                    ))}
                    
                     <View style={[styles.tableRow, { marginTop: 10, borderBottomWidth: 0 }]}>
                        <Text style={[styles.tableCell, { flex: 3, textAlign: 'right', fontFamily: 'Helvetica-Bold' }]}>TOTAL:</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'right', fontFamily: 'Helvetica-Bold' }]}> 
                            {data.tuitionItems?.reduce((acc, i) => acc + i.amount, 0).toFixed(2)}
                        </Text>
                    </View>
                     <View style={[styles.tableRow, { borderBottomWidth: 0 }]}>
                        <Text style={[styles.tableCell, { flex: 3, textAlign: 'right' }]}>PAID:</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'right' }]}> 
                             {data.tuitionItems?.reduce((acc, i) => acc + i.amount, 0).toFixed(2)}
                        </Text>
                    </View>
                     <View style={[styles.tableRow, { borderBottomWidth: 0 }]}>
                        <Text style={[styles.tableCell, { flex: 3, textAlign: 'right', fontFamily: 'Helvetica-Bold' }]}>BALANCE DUE:</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'right', fontFamily: 'Helvetica-Bold' }]}>0.00</Text>
                    </View>
                </View>



                <PageFooter />
            </Page>
        )}

        {/* PAGE 6: REGISTRATION RECEIPT */}
         {data.includeRegistrationReceipt && (
            <Page size={settings.paperSize} style={styles.page}>

                <Text style={styles.headerTitle}>Registration Acknowledgment</Text>
                <Text style={styles.headerSub}>Session: {data.termName}</Text>
                
                <Text style={styles.letterBody}>
                    This document acknowledges that <Text style={{ fontFamily: 'Times-Bold' }}>{data.firstName} {data.lastName}</Text> has successfully registered for the upcoming academic session.
                </Text>

                 <View style={[styles.table, { marginTop: 30 }]}>
                    <View style={[styles.tableRow, styles.tableHeader]}>
                         <Text style={[styles.tableCell, { flex: 3 }]}>Fee Description</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'right' }]}>Amount</Text>
                    </View>
                     {data.registrationItems?.length === 0 && (
                         <View style={styles.tableRow}>
                             <Text style={[styles.tableCell, { flex: 3 }]}>Registration & Processing Fee</Text>
                             <Text style={[styles.tableCell, { flex: 1, textAlign: 'right' }]}>50.00</Text>
                         </View>
                     )}
                     {data.registrationItems?.map((item) => (
                        <View key={item.id} style={styles.tableRow}>
                            <Text style={[styles.tableCell, { flex: 3 }]}>{item.description}</Text>
                            <Text style={[styles.tableCell, { flex: 1, textAlign: 'right' }]}>{item.amount.toFixed(2)}</Text>
                        </View>
                    ))}
                </View>
                
                <View style={{ marginTop: 20 }}>
                     <Text style={{ fontSize: 10, fontFamily: 'Times-Roman', color: '#666' }}>Processed by: {data.institutionName} — Bursar's Office</Text>
                     <Text style={{ fontSize: 10, fontFamily: 'Times-Roman', color: '#666' }}>Transaction Ref: {Math.random().toString(36).substr(2, 12).toUpperCase()}</Text>
                </View>

                <PageFooter />
            </Page>
        )}

        {/* PAGE 7: TRANSCRIPT */}
        {data.includeTranscript && (
            <Page size={settings.paperSize} style={styles.page}>

                <Text style={styles.headerTitle}>Academic Transcript - Unofficial</Text>
                <View style={{ flexDirection: 'row', justifyContent: 'space-between', marginBottom: 20 }}>
                    <Text style={styles.headerSub}>{data.institutionName}</Text>
                    <Text style={styles.headerSub}>Generated: {new Date().toLocaleDateString()}</Text>
                </View>
                
                <View style={[styles.table]}>
                    <View style={[styles.tableRow, styles.tableHeader]}>
                        <Text style={[styles.tableCell, { flex: 1 }]}>Term</Text>
                        <Text style={[styles.tableCell, { flex: 1 }]}>Course</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'center' }]}>Credits</Text>
                        <Text style={[styles.tableCell, { flex: 1, textAlign: 'center' }]}>Grade</Text>
                    </View>
                    {data.transcriptEntries?.map((entry) => (
                        <View key={entry.id} style={styles.tableRow}>
                             <Text style={[styles.tableCell, { flex: 1 }]}>{entry.term}</Text>
                             <Text style={[styles.tableCell, { flex: 1 }]}>{entry.course}</Text>
                             <Text style={[styles.tableCell, { flex: 1, textAlign: 'center' }]}>{entry.credits}</Text>
                             <Text style={[styles.tableCell, { flex: 1, textAlign: 'center' }]}>{entry.grade}</Text>
                        </View>
                    ))}
                     {(!data.transcriptEntries || data.transcriptEntries.length === 0) && (
                            <View style={styles.tableRow}>
                                <Text style={[styles.tableCell, { flex: 1 }]}>{data.termName}</Text>
                                <Text style={[styles.tableCell, { flex: 1 }]}>CS101 - Intro to CS</Text>
                                <Text style={[styles.tableCell, { flex: 1, textAlign: 'center' }]}>3</Text>
                                 <Text style={[styles.tableCell, { flex: 1, textAlign: 'center' }]}>IP</Text>
                            </View>
                     )}
                </View>

                 <View style={{ flexDirection: 'row', justifyContent: 'flex-end', marginTop: 10 }}>
                     <View style={{ width: 200 }}>
                         <View style={{ flexDirection: 'row', justifyContent: 'space-between', marginBottom: 5 }}>
                             <Text style={{ fontFamily: 'Helvetica-Bold', fontSize: 9 }}>Term GPA:</Text>
                              <Text style={{ fontSize: 9 }}>3.80</Text>
                         </View>
                         <View style={{ flexDirection: 'row', justifyContent: 'space-between' }}>
                             <Text style={{ fontFamily: 'Helvetica-Bold', fontSize: 9 }}>Cumulative GPA:</Text>
                              <Text style={{ fontSize: 9 }}>3.85</Text>
                         </View>
                     </View>
                 </View>



                 <View style={{ marginTop: 30, borderTopWidth: 1, borderTopColor: '#ccc', paddingTop: 10 }}>
                     <Text style={{ fontSize: 8, fontFamily: 'Helvetica-Bold', marginBottom: 3 }}>GRADING SYSTEM LEGEND:</Text>
                     <Text style={{ fontSize: 7, color: '#666', lineHeight: 1.3 }}>
                         A = 4.00 (Excellent); B = 3.00 (Good); C = 2.00 (Satisfactory); D = 1.00 (Passing); F = 0.00 (Failure).
                         I = Incomplete; W = Withdrawal; IP = In Progress.
                         One credit hour represents approximately 15 hours of class time per semester.
                     </Text>
                      <Text style={{ fontSize: 7, color: '#666', marginTop: 5, fontStyle: 'italic' }}>
                         This transcript is generated from the {data.institutionName} secure database. Alteration of this document may be a criminal offense.
                     </Text>
                 </View>

                <PageFooter />
            </Page>
        )}
        
        {/* PAGE 8: GOOD STANDING */}
        {data.includeGoodStandingLetter && (
            <Page size={settings.paperSize} style={styles.page}>

                 <View style={{ borderBottomWidth: 1, borderBottomColor: '#000', paddingBottom: 15, marginBottom: 20, alignItems: 'center' }}>
                    {data.institutionSeal && (
                         <Image src={data.institutionSeal} style={{ width: 60, height: 60, marginBottom: 10, objectFit: 'contain' }} />
                    )}
                     <Text style={{ fontSize: 18, fontFamily: 'Times-Bold', textTransform: 'uppercase' }}>{data.institutionName}</Text>
                </View>
                
                <Text style={{ fontSize: 11, fontFamily: 'Times-Bold', marginTop: 20, marginBottom: 10 }}>CERTIFICATE OF GOOD STANDING</Text>
                
                <Text style={styles.letterBody}>
                    This certifies that <Text style={{ fontFamily: 'Times-Bold' }}>{data.firstName} {data.lastName}</Text> is a student in good standing at {data.institutionName}.
                </Text>
                
                <Text style={styles.letterBody}>
                    The student has met all academic and conduct requirements for the current academic term and is eligible to continue their studies. There are no pending disciplinary actions or academic probations on their record.
                </Text>

                 <View style={styles.signatureBlock}>
                      <Text style={{ fontSize: 11, fontFamily: 'Times-Roman' }}>Sincerely,</Text>
                      <View style={{ height: 40 }} /> 
                      <Text style={{ fontSize: 11, fontFamily: 'Times-Bold' }}>Dean of Students</Text>
                      <Text style={{ fontSize: 11, fontFamily: 'Times-Roman' }}>{data.institutionName}</Text>
                 </View>


                <PageFooter />
            </Page>
        )}

        {/* FINAL: CERTIFICATE OF AUTHENTICITY */}
        <Page size={settings.paperSize} style={styles.page}>

             <View style={{ borderBottomWidth: 2, borderBottomColor: '#333', paddingBottom: 10, marginBottom: 30, alignItems: 'center' }}>
                 <Text style={{ fontSize: 22, fontFamily: 'Times-Bold', textTransform: 'uppercase' }}>Official Record Verification</Text>
                 <Text style={{ fontSize: 10, marginTop: 5, fontFamily: 'Times-Roman', color: '#666' }}>{data.institutionName} — Office of the Registrar</Text>
            </View>

            <View style={{ padding: 20, backgroundColor: '#f0fdf4', borderWidth: 1, borderColor: '#16a34a', marginBottom: 30 }}>
                <Text style={{ fontSize: 12, fontFamily: 'Times-Roman', textAlign: 'center', lineHeight: 1.6 }}>
                    This document certifies that the information contained herein is a true and accurate record of the student's academic history at <Text style={{ fontFamily: 'Times-Bold' }}>{data.institutionName}</Text>. This package is generated directly from our secured student records database.
                </Text>
            </View>

            <View style={[styles.table, { marginBottom: 30 }]}>
                <View style={styles.tableRow}>
                    <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Document Reference</Text>
                    <Text style={[styles.tableCell, { flex: 2, fontFamily: 'Helvetica-Bold' }]}>{systemRef}</Text>
                </View>
                 <View style={styles.tableRow}>
                    <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Date of Issuance</Text>
                    <Text style={[styles.tableCell, { flex: 2 }]}>{new Date().toISOString()}</Text>
                </View>
                 <View style={styles.tableRow}>
                    <Text style={[styles.tableCell, { flex: 1, color: '#666' }]}>Student Identifier</Text>
                    <Text style={[styles.tableCell, { flex: 2 }]}>{data.studentId}</Text>
                </View>
            </View>

            <View style={{ marginBottom: 20, paddingHorizontal: 40 }}>
                <Text style={{ fontSize: 9, fontFamily: 'Helvetica', textAlign: 'center', color: '#444' }}>
                    Valid document for fiscal and administrative procedures. Does not require a wet seal. Issued by CGRCE.
                </Text>
            </View>

            <View style={{ marginTop: 20, flexDirection: 'row', justifyContent: 'space-between', alignItems: 'flex-end' }}>
                <View style={{ alignItems: 'center', flex: 1 }}>
                     <View style={{ borderBottomWidth: 1, borderBottomColor: '#000', width: 200, marginBottom: 5, alignItems: 'center' }}>
                        <Text style={{ fontFamily: 'Times-Roman', fontSize: 16, marginBottom: 2, color: '#000080', fontStyle: 'italic' }}>
                            {data.registrarContact?.split('|')[0] || 'University Registrar'}
                        </Text>
                     </View>
                     <Text style={{ fontSize: 10, color: '#666' }}>Authorized Signature</Text>
                </View>
                <View style={{ alignItems: 'center', flex: 1 }}>
                     <View style={{ marginBottom: 5, alignItems: 'center', height: 40, justifyContent: 'center' }}>
                         {data.institutionSeal ? (
                             <Image src={data.institutionSeal} style={{ width: 40, height: 40, objectFit: 'contain' }} />
                         ) : (
                             <Text style={{ fontSize: 14, fontFamily: 'Times-Bold', color: '#333', textAlign: 'center' }}>{data.institutionName}</Text>
                         )}
                     </View>
                     <Text style={{ fontSize: 10, color: '#666' }}>Official Seal</Text>
                </View>
            </View>
             
            <View style={{ position: 'absolute', bottom: 100, left: 0, right: 0, alignItems: 'center' }}>
                 <Text style={{ fontSize: 8, color: '#ccc', fontFamily: 'Courier' }}>END OF PACKET - {systemRef}</Text>
            </View>

            <PageFooter />
        </Page>

      </Document>
    );
};
