import React, { useState, useEffect, useRef } from 'react';
import { Terminal, CheckCircle, AlertTriangle, Cpu, Play, Trash2, Globe, Activity } from 'lucide-react';

export const SheerIDBot = () => {
    const [verificationMode, setVerificationMode] = useState<'id' | 'link'>('id');
    const [requestID, setRequestID] = useState('');
    const [verificationLink, setVerificationLink] = useState('');
    const [isVerifying, setIsVerifying] = useState(false);
    const [verificationStatus, setVerificationStatus] = useState<'idle' | 'success' | 'failed'>('idle');
    const [logs, setLogs] = useState<string[]>([]);
    const logsEndRef = useRef<HTMLDivElement>(null);

    const scrollToBottom = () => {
        logsEndRef.current?.scrollIntoView({ behavior: "smooth" });
    };

    useEffect(() => {
        scrollToBottom();
    }, [logs]);

    const addLog = (message: string) => {
        setLogs(prev => [...prev, `[${new Date().toISOString().split('T')[1].split('.')[0]}] ${message}`]);
    };

    const handleVerify = async () => {
        let targetID = requestID;

        setVerificationStatus('idle');
        setLogs([]);
        setIsVerifying(true);
        addLog(`INITIALIZING SHEERID PROTOCOL v4.5.1...`);

        if (verificationMode === 'link') {
            if (!verificationLink) {
                addLog('ERROR: MISSING VERIFICATION LINK');
                setIsVerifying(false);
                return;
            }
            
            addLog(`ANALYZING URL STRUCTURE...`);
            await new Promise(r => setTimeout(r, 600));

            try {
                const urlObj = new URL(verificationLink);
                const idFromUrl = urlObj.searchParams.get('verificationId');
                
                if (!idFromUrl) {
                     addLog(`CRITICAL: INVALID LINK FORMAT.`);
                     addLog(`ERROR: PARAMETER 'verificationId' NOT FOUND.`);
                     setIsVerifying(false); 
                     setVerificationStatus('failed');
                     return;
                }
                
                targetID = idFromUrl;
                addLog(`URL VALID. EXTRACTED ID: ${targetID}`);
            } catch (e) {
                addLog(`CRITICAL: MALFORMED URL.`);
                setIsVerifying(false); 
                setVerificationStatus('failed');
                return;
            }
        } else {
             if (!requestID) {
                addLog('ERROR: MISSING VERIFICATION ID');
                setIsVerifying(false);
                return;
            }
        }
        
        // Simulation steps
        const steps = verificationMode === 'link' ? [
            { msg: `DETECTED ENCRYPTED STREAM...`, delay: 800 },
            { msg: `INITIATING BYPASS PROTOCOL (OVERRIDE-ALPHA)...`, delay: 1500 },
            { msg: `INJECTING AUTHENTICATION HEADERS...`, delay: 2400 },
            { msg: `BYPASSING FIREWALL RULES [###.......]`, delay: 3500 },
            { msg: `BYPASSING FIREWALL RULES [#######...]`, delay: 4200 },
            { msg: `BYPASSING FIREWALL RULES [##########]`, delay: 4800 },
            { msg: `ESTABLISHING SECURE TUNNEL TO TARGET: ${targetID}`, delay: 5900 },
            { msg: `FORCING SERVER HANDSHAKE...`, delay: 7000 },
        ] : [
            { msg: `CONNECTING TO SECURE GATEWAY (TLS 1.3)...`, delay: 800 },
            { msg: `CONNECTION ESTABLISHED. HANDSHAKE COMPLETE.`, delay: 1500 },
            { msg: `SEARCHING TARGET DATABASE FOR ID: ${requestID}...`, delay: 2400 },
            { msg: `TARGET ACQUIRED. ANALYZING METADATA...`, delay: 3500 },
            { msg: `VERIFYING INSTITUTIONAL CREDENTIALS...`, delay: 4800 },
            { msg: `CROSS-REFERENCING DOD DATABASE RECORDS...`, delay: 5900 },
            { msg: `VALIDATING SERVICE DATES...`, delay: 7000 },
        ];

        // In Link/Bypass mode, we never fail.
        let failed = verificationMode === 'link' ? false : Math.random() > 0.8; 

        for (const step of steps) {
            // No packet loss simulation in bypass mode to ensure speed
            if (verificationMode !== 'link' && Math.random() > 0.95) { 
                 addLog(`WARNING: PACKET LOSS DETECTED. RETRYING...`);
                 await new Promise(r => setTimeout(r, 1000));
            }
            await new Promise(r => setTimeout(r, step.delay - (step.delay * 0.5))); // Speed up a bit for UX
            addLog(step.msg);
        }

        await new Promise(r => setTimeout(r, 1000));

        if (failed) {
            addLog(`CRITICAL: VERIFICATION FAILED. STATUS CODE: 403`);
            addLog(`REASON: INSUFFICIENT DATA POINTS OR EXPIRED TOKEN.`);
            setVerificationStatus('failed');
        } else {
             if (verificationMode === 'link') {
                addLog(`SUCCESS: BYPASS SUCCESSFUL.`);
                addLog(`ACCESS GRANTED VIA OVERRIDE.`);
             } else {
                addLog(`SUCCESS: IDENTITY VERIFIED.`);
             }
            addLog(`GENERATING ASSERTION TOKEN...`);
             await new Promise(r => setTimeout(r, 800));
            addLog(`TOKEN GENERATED: ${Math.random().toString(36).substring(2, 15).toUpperCase()}`);
            setVerificationStatus('success');
        }

        setIsVerifying(false);
    };

    const clearConsole = () => {
        setLogs([]);
        setVerificationStatus('idle');
        setRequestID('');
        setVerificationLink('');
    };

    return (
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 h-[calc(100vh-8rem)]">
            {/* Control Panel */}
            <div className="lg:col-span-1 space-y-6">
                
                {/* Status Card */}
                <div className="bg-white dark:bg-slate-900 rounded-lg border border-slate-200 dark:border-slate-800 p-6 shadow-sm">
                    <div className="flex items-center space-x-3 mb-6">
                        <div className="p-2 bg-emerald-500/10 rounded-lg">
                            <Cpu className="w-6 h-6 text-emerald-500" />
                        </div>
                        <div>
                            <h2 className="text-lg font-bold text-slate-900 dark:text-white">Bot Status</h2>
                            <div className="flex items-center mt-1">
                                <div className={`w-2 h-2 rounded-full mr-2 ${isVerifying ? 'bg-amber-500 animate-pulse' : 'bg-emerald-500'}`}></div>
                                <span className="text-xs font-mono text-slate-500 uppercase">
                                    {isVerifying ? 'PROCESSING' : 'ONLINE / IDLE'}
                                </span>
                            </div>
                        </div>
                    </div>

                    <div className="space-y-4">
                        {/* Mode Toggle */}
                        <div className="grid grid-cols-2 gap-2 p-1 bg-slate-100 dark:bg-slate-950 rounded-lg">
                            <button
                                onClick={() => setVerificationMode('id')}
                                className={`text-xs font-medium py-1.5 rounded-md transition-all ${
                                    verificationMode === 'id' 
                                    ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' 
                                    : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                }`}
                            >
                                Request ID
                            </button>
                             <button
                                onClick={() => setVerificationMode('link')}
                                className={`text-xs font-medium py-1.5 rounded-md transition-all ${
                                    verificationMode === 'link' 
                                    ? 'bg-white dark:bg-slate-800 text-slate-900 dark:text-white shadow-sm' 
                                    : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                                }`}
                            >
                                Verify Link
                            </button>
                        </div>

                        {verificationMode === 'id' ? (
                            <div>
                                <label htmlFor="request-id" className="block text-xs font-mono font-medium text-slate-500 uppercase mb-2">
                                    Verification Request ID
                                </label>
                                <input 
                                    id="request-id"
                                    type="text" 
                                    value={requestID}
                                    onChange={(e) => setRequestID(e.target.value)}
                                    placeholder="REQ-0000-0000"
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-300 dark:border-slate-700 rounded-md px-4 py-2 text-sm font-mono text-slate-900 dark:text-slate-100 focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 outline-none transition-all placeholder:text-slate-400"
                                    disabled={isVerifying}
                                />
                            </div>
                        ) : (
                            <div>
                                <label htmlFor="verify-link" className="block text-xs font-mono font-medium text-slate-500 uppercase mb-2">
                                    Verification Link
                                </label>
                                <input 
                                    id="verify-link"
                                    type="text" 
                                    value={verificationLink}
                                    onChange={(e) => setVerificationLink(e.target.value)}
                                    placeholder="https://services.sheerid.com/..."
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-300 dark:border-slate-700 rounded-md px-4 py-2 text-sm font-mono text-slate-900 dark:text-slate-100 focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 outline-none transition-all placeholder:text-slate-400"
                                    disabled={isVerifying}
                                />
                                <p className="text-[10px] text-slate-400 mt-1.5">
                                    System will extract <code>verificationId</code> from URL parameters.
                                </p>
                            </div>
                        )}


                        <div className="grid grid-cols-2 gap-3">
                            <button
                                onClick={handleVerify}
                                disabled={isVerifying || (verificationMode === 'id' ? !requestID : !verificationLink)}
                                className={`flex items-center justify-center py-2 px-4 rounded-md text-sm font-medium transition-all ${
                                    isVerifying || (verificationMode === 'id' ? !requestID : !verificationLink)
                                        ? 'bg-slate-100 dark:bg-slate-800 text-slate-400 cursor-not-allowed'
                                        : 'bg-emerald-500 hover:bg-emerald-600 text-white shadow-lg shadow-emerald-500/20'
                                }`}
                            >
                                <Play className="w-4 h-4 mr-2" />
                                {isVerifying ? 'Running...' : 'Run Verify'}
                            </button>
                            <button
                                onClick={clearConsole}
                                disabled={isVerifying}
                                className="flex items-center justify-center py-2 px-4 rounded-md text-sm font-medium bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 text-slate-700 dark:text-slate-300 hover:bg-slate-50 dark:hover:bg-slate-700 transition-all"
                            >
                                <Trash2 className="w-4 h-4 mr-2" />
                                Clear
                            </button>
                        </div>
                    </div>
                </div>

                {/* Network Stats (Decorative) */}
                <div className="bg-white dark:bg-slate-900 rounded-lg border border-slate-200 dark:border-slate-800 p-6 shadow-sm overflow-hidden relative">
                     <div className="flex items-center justify-between mb-4">
                        <h3 className="text-xs font-bold font-mono text-slate-500 uppercase tracking-wider">Network Metrics</h3>
                        <Activity className="w-4 h-4 text-slate-400" />
                    </div>
                    
                    <div className="space-y-4">
                         <div className="flex justify-between items-center text-sm">
                            <span className="text-slate-400">Latency</span>
                            <span className="font-mono text-emerald-500">24ms</span>
                         </div>
                         <div className="w-full bg-slate-100 dark:bg-slate-800 h-1.5 rounded-full overflow-hidden">
                            <div className="h-full bg-emerald-500 w-[24%]"></div>
                         </div>

                         <div className="flex justify-between items-center text-sm">
                            <span className="text-slate-400">Success Rate</span>
                            <span className="font-mono text-emerald-500">99.8%</span>
                         </div>
                         <div className="w-full bg-slate-100 dark:bg-slate-800 h-1.5 rounded-full overflow-hidden">
                            <div className="h-full bg-emerald-500 w-[99%]"></div>
                         </div>
                         
                         <div className="flex justify-between items-center text-sm">
                            <span className="text-slate-400">API Quota</span>
                            <span className="font-mono text-blue-500">450/1000</span>
                         </div>
                         <div className="w-full bg-slate-100 dark:bg-slate-800 h-1.5 rounded-full overflow-hidden">
                            <div className="h-full bg-blue-500 w-[45%]"></div>
                         </div>
                    </div>
                </div>

            </div>

             {/* Terminal Output */}
            <div className="lg:col-span-2 flex flex-col h-full bg-slate-950 rounded-lg border border-slate-800 font-mono text-sm shadow-2xl overflow-hidden relative">
                
                {/* Terminal Header */}
                <div className="flex items-center justify-between px-4 py-2 bg-slate-900 border-b border-slate-800">
                    <div className="flex items-center space-x-2">
                        <Terminal className="w-4 h-4 text-slate-400" />
                        <span className="text-slate-300 text-xs font-semibold">SHEERID_PROTOCOL_EXEC.sh</span>
                    </div>
                     <div className="flex space-x-1.5">
                        <div className="w-2.5 h-2.5 rounded-full bg-slate-700"></div>
                        <div className="w-2.5 h-2.5 rounded-full bg-slate-700"></div>
                        <div className="w-2.5 h-2.5 rounded-full bg-slate-700"></div>
                    </div>
                </div>

                {/* Terminal Body */}
                <div className="flex-1 p-4 overflow-y-auto space-y-1 scrollbar-thin scrollbar-thumb-slate-700 scrollbar-track-transparent">
                     {logs.length === 0 && (
                        <div className="h-full flex flex-col items-center justify-center text-slate-600 opacity-50">
                            <Globe className="w-12 h-12 mb-4 animate-pulse" />
                            <p>AWAITING COMMAND INPUT...</p>
                        </div>
                     )}
                     
                     {logs.map((log, index) => {
                        let colorClass = 'text-slate-300';
                        if (log.includes('ERROR') || log.includes('FAILED')) colorClass = 'text-red-400';
                        else if (log.includes('SUCCESS')) colorClass = 'text-emerald-400';
                        else if (log.includes('WARNING')) colorClass = 'text-amber-400';

                        return (
                            <div key={`${index}-${log.substring(0, 5)}`} className={`font-mono ${colorClass}`}>
                                {log}
                            </div>
                        );
                    })}
                    <div ref={logsEndRef} />
                </div>

                {/* Overlay Result */}
                 {verificationStatus !== 'idle' && (
                    <div className="absolute inset-0 bg-slate-950/70 backdrop-blur-sm flex items-center justify-center z-10 animate-in fade-in duration-300">
                        <div className={`transform scale-100 transition-all duration-300 p-8 rounded-xl border-2 flex flex-col items-center shadow-2xl ${verificationStatus === 'success' ? 'bg-emerald-950/90 border-emerald-500' : 'bg-red-950/90 border-red-500'}`}>
                            {verificationStatus === 'success' ? (
                                <>
                                    <div className="w-16 h-16 rounded-full bg-emerald-500/20 flex items-center justify-center mb-4 border border-emerald-500">
                                        <CheckCircle className="w-8 h-8 text-emerald-500" />
                                    </div>
                                    <h3 className="text-2xl font-bold text-white tracking-wider mb-1">VERIFIED</h3>
                                    <p className="text-emerald-400 font-mono text-sm">IDENTITY CONFIRMED</p>
                                    <div className="mt-6 px-4 py-1.5 bg-emerald-500/10 rounded border border-emerald-500/30 text-emerald-300 text-xs font-mono">
                                        ID: {Math.random().toString(16).substring(2, 10).toUpperCase()}
                                    </div>
                                </>
                            ) : (
                                <>
                                      <div className="w-16 h-16 rounded-full bg-red-500/20 flex items-center justify-center mb-4 border border-red-500">
                                        <AlertTriangle className="w-8 h-8 text-red-500" />
                                    </div>
                                    <h3 className="text-2xl font-bold text-white tracking-wider mb-1">DENIED</h3>
                                    <p className="text-red-400 font-mono text-sm">VERIFICATION FAILED</p>
                                     <div className="mt-6 px-4 py-1.5 bg-red-500/10 rounded border border-red-500/30 text-red-300 text-xs font-mono">
                                        CODE: ERR_VERIFY_TIMEOUT
                                    </div>
                                </>
                            )}
                            
                            <button 
                                onClick={(e) => {
                                    e.stopPropagation();
                                    setVerificationStatus('idle');
                                }}
                                className="mt-8 text-xs text-slate-400 hover:text-white underline underline-offset-4"
                            >
                                CLOSE RESULT
                            </button>
                        </div>
                    </div>
                )}
            </div>
        </div>
    );
};
