import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/Card';
import { Input } from '@/components/ui/Input';
import { ServiceRecord, RecordType } from '@/types/records';
import { FileText, Search, Eye, Trash2, Award, GraduationCap } from 'lucide-react';
import { RecordDetailModal } from './RecordDetailModal';

interface ServiceRecordsIndexProps {
  records: ServiceRecord[];
  onSearch: (query: string, type?: RecordType) => ServiceRecord[];
  onLoadDraft: (record: ServiceRecord) => void;
  onDelete: (id: string) => void;
}

export const ServiceRecordsIndex: React.FC<ServiceRecordsIndexProps> = ({
  records,
  onSearch,
  onLoadDraft,
  onDelete
}) => {
  const [searchQuery, setSearchQuery] = useState('');
  const [activeFilter, setActiveFilter] = useState<RecordType | undefined>(undefined);
  const [selectedRecord, setSelectedRecord] = useState<ServiceRecord | null>(null);

  const filteredRecords = onSearch(searchQuery, activeFilter);

  const getBadgeColor = (type: RecordType) => {
    return type === 'MILITARY' 
      ? 'bg-blue-100 text-blue-800 border-blue-200 dark:bg-blue-900/30 dark:text-blue-300 dark:border-blue-800'
      : 'bg-indigo-100 text-indigo-800 border-indigo-200 dark:bg-indigo-900/30 dark:text-indigo-300 dark:border-indigo-800';
  };

  const getBadgeIcon = (type: RecordType) => {
    return type === 'MILITARY' ? <Award className="w-3 h-3 mr-1" /> : <GraduationCap className="w-3 h-3 mr-1" />;
  };

  return (
    <Card className="border-slate-800 bg-slate-900/50">
      <CardHeader className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4 border-b border-slate-800 pb-4">
        <div>
          <CardTitle className="text-lg font-medium text-slate-100 flex items-center">
            <FileText className="w-5 h-5 mr-2 text-slate-400" />
            SERVICE RECORD INDEX
          </CardTitle>
          <p className="text-xs text-slate-500 font-mono mt-1">GLOBAL ARCHIVE CONTROL</p>
        </div>
        
        <div className="flex flex-col sm:flex-row gap-2 w-full sm:w-auto">
          <div className="relative">
            <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-slate-500" />
            <Input 
              placeholder="Search Ref, Name, Inst..." 
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-9 w-full sm:w-64 bg-slate-800 border-slate-700 text-slate-200 text-sm focus:ring-slate-500"
            />
          </div>
          <div className="flex bg-slate-800 rounded-md p-1 border border-slate-700">
            <button
              onClick={() => setActiveFilter(undefined)}
              className={`px-3 py-1.5 text-xs font-medium rounded transition-colors ${activeFilter === undefined ? 'bg-slate-700 text-white shadow-sm' : 'text-slate-400 hover:text-slate-200'}`}
            >
              ALL
            </button>
            <button
              onClick={() => setActiveFilter('MILITARY')}
              className={`px-3 py-1.5 text-xs font-medium rounded transition-colors flex items-center ${activeFilter === 'MILITARY' ? 'bg-blue-900/50 text-blue-200 shadow-sm' : 'text-slate-400 hover:text-slate-200'}`}
            >
              <Award className="w-3 h-3 mr-1" /> MIL
            </button>
            <button
              onClick={() => setActiveFilter('STUDENT')}
              className={`px-3 py-1.5 text-xs font-medium rounded transition-colors flex items-center ${activeFilter === 'STUDENT' ? 'bg-indigo-900/50 text-indigo-200 shadow-sm' : 'text-slate-400 hover:text-slate-200'}`}
            >
              <GraduationCap className="w-3 h-3 mr-1" /> STU
            </button>
          </div>
        </div>
      </CardHeader>
      
      <CardContent className="p-0">
        <div className="overflow-x-auto">
          <table className="w-full text-sm text-left">
            <thead className="text-xs text-slate-400 uppercase bg-slate-900/80 border-b border-slate-800">
              <tr>
                <th className="px-4 py-3 w-32">Type</th>
                <th className="px-4 py-3">System Ref</th>
                <th className="px-4 py-3">Subject Name</th>
                <th className="px-4 py-3">Origin</th>
                <th className="px-4 py-3">Date</th>
                <th className="px-4 py-3 text-right">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-slate-800">
              {filteredRecords.length > 0 ? (
                filteredRecords.map((record) => (
                  <tr key={record.id} className="hover:bg-slate-800/50 transition-colors group">
                    <td className="px-4 py-3">
                      <span className={`inline-flex items-center px-2 py-0.5 rounded text-[10px] font-medium border ${getBadgeColor(record.type)}`}>
                        {getBadgeIcon(record.type)}
                        {record.type}
                      </span>
                    </td>
                    <td className="px-4 py-3 font-mono text-xs text-slate-400">
                      {record.systemRef}
                    </td>
                    <td className="px-4 py-3 font-medium text-slate-200">
                      {record.details.name}
                    </td>
                    <td className="px-4 py-3 text-slate-400 text-xs">
                      {record.type === 'MILITARY' 
                        ? (record as any).details.branch 
                        : (record as any).details.institution
                      }
                    </td>
                    <td className="px-4 py-3 text-slate-500 text-xs font-mono">
                      {new Date(record.generatedAt).toLocaleDateString()}
                    </td>
                    <td className="px-4 py-3 text-right space-x-2 opacity-60 group-hover:opacity-100 transition-opacity">
                      <button 
                        onClick={() => setSelectedRecord(record)}
                        className="text-slate-400 hover:text-blue-400 transition-colors p-1"
                        title="View Record Details"
                      >
                        <Eye size={16} />
                      </button>
                      <button 
                        onClick={() => onDelete(record.id)}
                        className="text-slate-400 hover:text-red-400 transition-colors p-1"
                        title="Archive Record"
                      >
                        <Trash2 size={16} />
                      </button>
                    </td>
                  </tr>
                ))
              ) : (
                <tr>
                  <td colSpan={6} className="px-4 py-8 text-center text-slate-500 text-xs italic">
                    NO RECORDS FOUND MATCHING QUERY
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      </CardContent>

      {selectedRecord && (
        <RecordDetailModal 
          record={selectedRecord} 
          onClose={() => setSelectedRecord(null)} 
          onLoadAsDraft={(r) => {
            onLoadDraft(r);
            setSelectedRecord(null);
          }}
        />
      )}
    </Card>
  );
};
