"use client";

import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/Card';
import { Label } from '@/components/ui/Label';
import { Input } from '@/components/ui/Input';
import { Select } from '@/components/ui/Select';
import { Button } from '@/components/ui/Button';
import { DateSelect } from '@/components/DateSelect';
import { STATUS_OPTIONS, BRANCH_OPTIONS } from '@/lib/constants';
import { validateVerificationData } from '@/lib/validator';
import { generateProfile, VerificationData } from '@/lib/generator';
import { RefreshCw, Loader2, Download, FileText, User, Shield, Camera, Settings } from 'lucide-react';
import { pdf } from '@react-pdf/renderer';
import { VectorPdfDocument } from './VectorPdfDocument';
import { SystemTerminal } from './SystemTerminal';
import { useServiceRecords } from '@/hooks/useServiceRecords';
import { ServiceRecordsIndex } from './ServiceRecordsIndex';
import { TemplateManager } from './TemplateManager';
import { MilitaryUnitTemplate } from '@/types/templates';
import { MilitaryRecord, ServiceRecord } from '@/types/records';

// Log definition
interface LogEntry {
  id: string;
  timestamp: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error';
}

export function RecordEditor() {
  const [formData, setFormData] = useState<Partial<VerificationData>>({});
  const [isGenerating, setIsGenerating] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [successMessage, setSuccessMessage] = useState<string | null>(null);
  const [logs, setLogs] = useState<LogEntry[]>([]);

  const { addRecord, records, searchRecords, deleteRecord } = useServiceRecords();

  const handleApplyUnitTemplate = (data: MilitaryUnitTemplate['data']) => {
      const updates: Partial<VerificationData> = {};
      if (data.branch) updates.branch = data.branch;
      if (data.recruitmentCenter) updates.recruitmentCenter = data.recruitmentCenter;
      if (data.officialName) updates.authorizedOfficialName = data.officialName;
      // Unit name and base address might need to map to specific fields or notes if not present in VerificationData
      // VerificationData has "recruitmentCenter", "branch".
      // Let's assume Unit Name goes to "notes" or is just for the template reference?
      // Or maybe add "unitName" to VerificationData? For now, we utilize what we have.
      
      setFormData(prev => ({ ...prev, ...updates }));
      addLog(`UNIT TEMPLATE APPLIED: ${data.unitName.toUpperCase()}`, 'success');
  };

  const getCurrentUnitState = (): MilitaryUnitTemplate['data'] => {
      // Best effort map
      return {
          branch: formData.branch || 'Army',
          unitName: formData.recruitmentCenter || 'Unknown Unit', // Fallback
          recruitmentCenter: formData.recruitmentCenter,
          officialName: formData.authorizedOfficialName
      };
  };

  const addLog = (message: string, type: LogEntry['type'] = 'info') => {
      const now = new Date();
      const timeString = now.toLocaleTimeString('en-US', { hour12: false });
      const newLog: LogEntry = {
          id: Math.random().toString(36).substr(2, 9),
          timestamp: timeString,
          message,
          type
      };
      setLogs(prev => [...prev, newLog]);
  };

  const handleLoadDraft = (record: ServiceRecord) => {
    if (record.type === 'MILITARY') {
        setFormData(record.data);
        addLog(`LOADED RECORD: ${record.details.name} (REF: ${record.systemRef})`, 'info');
        setSuccessMessage('Draft loaded from history.');
    } else {
        alert('Cannot load Student record into Military editor.');
    }
  };

  const handleAutofill = () => {
    setIsGenerating(true);
    setSuccessMessage(null);
    addLog('INITIATING AUTO-POPULATION SEQUENCE...', 'info');
    
    setTimeout(() => {
      const data = generateProfile();
      setFormData(data);
      setErrors({});
      setIsGenerating(false);
      addLog('IDENTITY FOUND. POPULATING FIELDS...', 'success');
      addLog(`LOADED RECORD: ${data.lastName?.toUpperCase()}, ${data.firstName?.toUpperCase()}`, 'info');
    }, 800);
  };

  const handleClear = () => {
    setFormData({});
    setErrors({});
    setSuccessMessage(null);
    addLog('FORM DATA CLEARED. READY FOR INPUT.', 'warning');
  };

  const handleChange = (field: keyof VerificationData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    if (errors[field]) {
      setErrors(prev => {
        const newErrors = { ...prev };
        delete newErrors[field];
        return newErrors;
      });
    }
  };

  const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>, field: 'logo' | 'photo') => {
      const file = event.target.files?.[0];
      if (file) {
          addLog(`UPLOADING ASSET: ${file.name.toUpperCase()}...`, 'info');
          const reader = new FileReader();
          reader.onloadend = () => {
              handleChange(field, reader.result as string);
              addLog(`ASSET UPLOAD COMPLETE: ${field.toUpperCase()}`, 'success');
          };
          reader.readAsDataURL(file);
      }
  };

  const handleDownload = async () => {
    addLog('VALIDATING RECORD INTEGRITY...', 'info');
    const newErrors = validateVerificationData(formData);
    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors);
      addLog(`VALIDATION FAILED: ${Object.keys(newErrors).length} ERROR(S) DETECTED.`, 'error');
      return;
    }

    setIsGenerating(true);
    addLog('COMPILING VECTOR PDF DOCUMENT...', 'info');
    
    try {
        const blob = await pdf(<VectorPdfDocument data={formData} />).toBlob();
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        const filename = `Service_Record_${formData.lastName || 'Unknown'}_${new Date().getTime()}.pdf`;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        link.remove();
        URL.revokeObjectURL(url);
        
        setIsGenerating(false);
        setSuccessMessage('Record exported successfully.');
        addLog(`EXPORT SUCCESSFUL: ${filename}`, 'success');
        addLog(`EXPORT SUCCESSFUL: ${filename}`, 'success');
        addLog('SECURE CONNECTION CLOSED.', 'info');

        // Save to Global History
        const sysRef = `MIL-${new Date().getFullYear()}-${Math.floor(Math.random() * 10000)}`;
        const newRecord: MilitaryRecord = {
            id: Math.random().toString(36).substr(2, 9),
            systemRef: sysRef,
            generatedAt: new Date().toISOString(),
            type: 'MILITARY',
            status: 'GENERATED',
            isLocked: true,
            details: {
                name: `${formData.lastName}, ${formData.firstName}`,
                branch: formData.branch || 'UNKNOWN',
                rank: formData.rank || 'N/A',
                dischargeDate: formData.dischargeDate?.toISOString()
            },
            data: formData as VerificationData,
        };
        addRecord(newRecord);
        addLog(`RECORD ARCHIVED: ${sysRef}`, 'success');
        
    } catch (err: any) {
        console.error('PDF Generation Error:', err);
        setIsGenerating(false);
        addLog(`SYSTEM ERROR: ${err.message || 'UNKNOWN FATAL ERROR'}`, 'error');
        alert(`System Error: ${err.message || err.toString()}`);
    }
  };

  return (
    <div className="space-y-6">
      
      <TemplateManager 
          type="MILITARY_UNIT" 
          label="Unit / Base"
          onApply={handleApplyUnitTemplate}
          currentData={getCurrentUnitState()}
      />

      {/* Top Action Bar */}
      <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4 bg-white dark:bg-slate-900 p-4 rounded-lg border border-slate-200 dark:border-slate-800 shadow-sm">
        <div>
            <h2 className="text-lg font-bold text-slate-800 dark:text-slate-100 flex items-center">
                <FileText className="w-5 h-5 mr-2 text-emerald-600" />
                DD-214 GENERATOR
            </h2>
            <p className="text-xs text-slate-500 font-mono mt-1" suppressHydrationWarning>SYSTEM REF: {new Date().getFullYear()}-GK-{Math.floor(Math.random() * 1000)}</p>
        </div>
        <div className="flex space-x-3">
             <Button variant="secondary" onClick={handleClear} size="sm" className="text-xs border-slate-300">
                CLEAR FORM
            </Button>
            <Button onClick={handleAutofill} disabled={isGenerating} size="sm" className="bg-slate-800 hover:bg-slate-700 text-white text-xs">
                <RefreshCw className={`mr-2 h-3 w-3 ${isGenerating ? 'animate-spin' : ''}`} />
                AUTO-POPULATE
            </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        
        {/* Left Col: Personal & Service Data */}
        <div className="lg:col-span-2 space-y-6">
            
            {/* Identity Card */}
            <Card className="border-t-4 border-t-emerald-500 shadow-sm">
                <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                    <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                        <User className="w-4 h-4 mr-2" /> 
                        Identity Information
                    </CardTitle>
                </CardHeader>
                <CardContent className="pt-6 grid gap-6 sm:grid-cols-2">
                     <div className="space-y-1.5">
                        <Label className="text-xs uppercase text-slate-500 font-semibold" required>First Name</Label>
                        <Input 
                            value={formData.firstName || ''} 
                            onChange={(e) => handleChange('firstName', e.target.value)}
                            error={!!errors.firstName}
                            className="font-mono text-sm bg-slate-50"
                            placeholder="JOHN"
                        />
                    </div>
                    <div className="space-y-1.5">
                        <Label className="text-xs uppercase text-slate-500 font-semibold" required>Last Name</Label>
                         <Input 
                            value={formData.lastName || ''} 
                            onChange={(e) => handleChange('lastName', e.target.value)}
                            error={!!errors.lastName}
                            className="font-mono text-sm bg-slate-50"
                            placeholder="DOE"
                        />
                    </div>
                     <div className="space-y-1.5 sm:col-span-2">
                        <Label className="text-xs uppercase text-slate-500 font-semibold" required>Date of Birth</Label>
                        <DateSelect 
                            value={formData.dob} 
                            onChange={(date) => handleChange('dob', date)}
                            startYear={1920}
                            endYear={new Date().getFullYear() - 15}
                        />
                    </div>
                </CardContent>
            </Card>

            {/* Service Details Card */}
             <Card className="border-t-4 border-t-blue-500 shadow-sm">
                <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                    <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                        <Shield className="w-4 h-4 mr-2" /> 
                        Service Record
                    </CardTitle>
                </CardHeader>
                <CardContent className="pt-6 space-y-4">
                     <div className="grid gap-4 sm:grid-cols-2">
                        <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold" required>Branch</Label>
                            <Select 
                                value={formData.branch || ''} 
                                onChange={(e) => handleChange('branch', e.target.value)}
                                error={!!errors.branch}
                            >
                                <option value="" disabled>SELECT BRANCH</option>
                                {BRANCH_OPTIONS.map(opt => <option key={opt} value={opt}>{opt}</option>)}
                            </Select>
                        </div>
                        <div className="space-y-1.5">
                            <Label className="text-xs uppercase text-slate-500 font-semibold" required>Status</Label>
                            <Select 
                                value={formData.status || ''} 
                                onChange={(e) => handleChange('status', e.target.value)}
                                error={!!errors.status}
                            >
                                <option value="" disabled>SELECT STATUS</option>
                                {STATUS_OPTIONS.map(opt => <option key={opt} value={opt}>{opt}</option>)}
                            </Select>
                        </div>
                    </div>

                    <div className="grid gap-4 sm:grid-cols-2">
                         <div className="space-y-1.5">
                             <Label className="text-xs uppercase text-slate-500 font-semibold" required>Service Start</Label>
                             <DateSelect 
                                value={formData.startDate} 
                                onChange={(date) => handleChange('startDate', date)}
                                startYear={1950}
                             />
                         </div>
                         <div className="space-y-1.5">
                             <Label className="text-xs uppercase text-slate-500 font-semibold" required>Discharge Date</Label>
                             <DateSelect 
                                value={formData.dischargeDate} 
                                onChange={(date) => handleChange('dischargeDate', date)}
                                startYear={1950}
                                endYear={new Date().getFullYear() + 20}
                             />
                         </div>
                    </div>

                    <div className="space-y-1.5">
                         <Label className="text-xs uppercase text-slate-500 font-semibold" required>Recruitment Center (MEPS)</Label>
                         <Input 
                            value={formData.recruitmentCenter || ''} 
                            onChange={(e) => handleChange('recruitmentCenter', e.target.value)}
                            error={!!errors.recruitmentCenter}
                            className="font-mono text-sm"
                            placeholder="e.g. CHICAGO MEPS"
                        />
                    </div>
                </CardContent>
             </Card>
        </div>

        {/* Right Col: Admin & Actions */}
        <div className="space-y-6 flex flex-col">
             {/* Admin Card */}
             <Card className="border-t-4 border-t-slate-500 shadow-sm">
                <CardHeader className="pb-3 border-b border-gray-100 bg-slate-50/50">
                    <CardTitle className="text-sm font-bold uppercase tracking-wider text-slate-600 flex items-center">
                        <Settings className="w-4 h-4 mr-2" /> 
                        Admin Controls
                    </CardTitle>
                </CardHeader>
                <CardContent className="pt-6 space-y-4">
                     <div className="space-y-1.5">
                        <Label className="text-xs uppercase text-slate-500 font-semibold" required>Contact Email</Label>
                        <Input 
                           value={formData.email || ''} 
                           onChange={(e) => handleChange('email', e.target.value)}
                           error={!!errors.email}
                           className="font-mono text-sm"
                           placeholder="user@mil.gov"
                        />
                    </div>
                    
                    <div className="pt-4 border-t border-dashed border-gray-200">
                         <Label className="text-xs uppercase text-slate-500 font-semibold mb-2 block">Custom Assets</Label>
                         
                         <div className="grid grid-cols-2 gap-2">
                             <div className="border border-slate-200 rounded p-2 text-center hover:bg-slate-50 transition-colors cursor-pointer relative overflow-hidden group">
                                 <input type="file" className="absolute inset-0 opacity-0 cursor-pointer" onChange={(e) => handleImageUpload(e, 'logo')} accept="image/*" />
                                 <Shield className="w-6 h-6 mx-auto text-slate-400 mb-1 group-hover:text-emerald-500" />
                                 <span className="text-[10px] uppercase font-bold text-slate-500">Upload Seal</span>
                             </div>
                             <div className="border border-slate-200 rounded p-2 text-center hover:bg-slate-50 transition-colors cursor-pointer relative overflow-hidden group">
                                 <input type="file" className="absolute inset-0 opacity-0 cursor-pointer" onChange={(e) => handleImageUpload(e, 'photo')} accept="image/*" />
                                 <Camera className="w-6 h-6 mx-auto text-slate-400 mb-1 group-hover:text-blue-500" />
                                  <span className="text-[10px] uppercase font-bold text-slate-500">Member Photo</span>
                             </div>
                         </div>
                    </div>
                </CardContent>
             </Card>

              {/* Status & Actions */}
             <div className="bg-slate-900 rounded-lg p-4 text-white shadow-lg">
                <div className="mb-4">
                    <h3 className="text-xs font-mono text-emerald-400 mb-1">SYSTEM STATUS</h3>
                    <div className="flex items-center space-x-2">
                        <div className="w-2 h-2 rounded-full bg-emerald-500 animate-pulse"></div>
                        <span className="text-sm font-bold tracking-wider">READY FOR PROCESSING</span>
                    </div>
                </div>
                
                {successMessage && (
                    <div className="mb-4 p-2 bg-emerald-900/50 border border-emerald-500/30 rounded text-xs text-emerald-300 font-mono">
                         {'>'} {successMessage}
                    </div>
                )}

                <Button 
                    className="w-full bg-emerald-600 hover:bg-emerald-500 text-white font-bold tracking-wider" 
                    size="lg" 
                    onClick={handleDownload} 
                    disabled={isGenerating}
                >
                    {isGenerating ? (
                        <span className="flex items-center"><Loader2 className="mr-2 h-4 w-4 animate-spin" /> PROCESSING...</span>
                    ) : (
                        <span className="flex items-center justify-center"><Download className="mr-2 h-4 w-4" /> GENERATE RECORD</span>
                    )}
                </Button>
             </div>

             {/* System Terminal - Now integrated here */}
             <SystemTerminal logs={logs} />
        </div>

      </div>
      
      {/* History Section */}
      <div className="bg-slate-50 dark:bg-slate-900 p-6 rounded-lg border-2 border-slate-200 dark:border-slate-800 border-dashed">
          <div className="mb-4 flex items-center justify-between">
              <h3 className="text-sm font-bold text-slate-500 uppercase tracking-wider">Session History</h3>
          </div>
          <ServiceRecordsIndex 
            records={records}
            onSearch={searchRecords}
            onLoadDraft={handleLoadDraft}
            onDelete={deleteRecord}
          />
      </div>

    </div>
  );
}
