import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle, CardFooter } from '@/components/ui/Card';
import { Button } from '@/components/ui/Button';
import { ServiceRecord } from '@/types/records';
import { FileText, Download, X, Copy, RefreshCw, Award, GraduationCap } from 'lucide-react';
import { pdf } from '@react-pdf/renderer';
import { VectorPdfDocument } from './VectorPdfDocument';
import { StudentPacketPdf } from './StudentPacketPdf';

interface RecordDetailModalProps {
  record: ServiceRecord;
  onClose: () => void;
  onLoadAsDraft: (record: ServiceRecord) => void;
}

export const RecordDetailModal: React.FC<RecordDetailModalProps> = ({ record, onClose, onLoadAsDraft }) => {
  const [isRegenerating, setIsRegenerating] = useState(false);

  const handleRegeneratePdf = async () => {
    setIsRegenerating(true);
    try {
      let blob: Blob;
      let filename = '';

      if (record.type === 'MILITARY') {
         blob = await pdf(<VectorPdfDocument data={record.data} />).toBlob();
         filename = `Copy_Service_Record_${record.details.name}_${record.systemRef}.pdf`;
      } else {
         blob = await pdf(
             <StudentPacketPdf 
                data={record.data} 
                settings={record.pdfSettings}
                // default ID settings for regeneration if not stored
                idSettings={{ theme: 'LIGHT', barcodeType: 'QR', expirationMode: 'AUTO' }} 
                systemRef={record.systemRef} 
            />
         ).toBlob();
         filename = `Copy_Student_Packet_${record.details.name}_${record.systemRef}.pdf`;
      }

      const url = URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      link.download = filename;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error("Regeneration failed", e);
      alert("Failed to regenerate PDF");
    } finally {
      setIsRegenerating(false);
    }
  };

  const handleExportJson = () => {
    const dataStr = JSON.stringify(record, null, 2);
    const blob = new Blob([dataStr], { type: "application/json" });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `record_${record.systemRef}.json`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50 backdrop-blur-sm p-4">
      <Card className="w-full max-w-2xl bg-white dark:bg-slate-900 border-slate-200 dark:border-slate-800 shadow-2xl max-h-[90vh] flex flex-col">
        <CardHeader className="flex flex-row items-center justify-between border-b border-slate-100 dark:border-slate-800 pb-4">
          <div className="flex items-center space-x-3">
            <div className={`p-2 rounded-full ${record.type === 'MILITARY' ? 'bg-blue-100 text-blue-600' : 'bg-indigo-100 text-indigo-600'}`}>
               {record.type === 'MILITARY' ? <Award className="w-6 h-6" /> : <GraduationCap className="w-6 h-6" />}
            </div>
            <div>
              <CardTitle className="text-lg font-bold text-slate-800 dark:text-slate-100">
                {record.type === 'MILITARY' ? 'MILITARY SERVICE RECORD' : 'STUDENT VERIFICATION PACKET'}
              </CardTitle>
              <p className="text-xs text-slate-500 font-mono">REF: {record.systemRef}</p>
            </div>
          </div>
          <Button variant="ghost" size="sm" onClick={onClose} className="h-8 w-8 p-0 rounded-full">
            <X className="w-5 h-5" />
          </Button>
        </CardHeader>
        
        <CardContent className="flex-1 overflow-y-auto p-6 space-y-6">
           <div className="grid grid-cols-2 gap-4">
             <div className="space-y-1">
               <span className="text-xs font-bold text-slate-400 uppercase">Subject Name</span>
               <p className="font-medium text-slate-800 dark:text-slate-200 text-sm">{record.details.name}</p>
             </div>
             <div className="space-y-1">
               <span className="text-xs font-bold text-slate-400 uppercase">Generated Date</span>
               <p className="font-mono text-slate-800 dark:text-slate-200 text-sm">{new Date(record.generatedAt).toLocaleString()}</p>
             </div>
             <div className="space-y-1">
               <p className="font-medium text-slate-800 dark:text-slate-200 text-sm">
                 {record.type === 'STUDENT' ? record.details.institution : record.details.branch}
               </p>
             </div>
             <div className="space-y-1">
               <span className="text-xs font-bold text-slate-400 uppercase">Reference ID</span>
               <p className="font-mono text-slate-800 dark:text-slate-200 text-sm bg-slate-100 dark:bg-slate-800 inline-block px-2 py-0.5 rounded">
                 {record.id}
               </p>
             </div>
           </div>

           <div className="bg-slate-50 dark:bg-slate-800/50 p-4 rounded-lg border border-slate-100 dark:border-slate-800">
              <h4 className="text-xs font-bold text-slate-500 uppercase mb-3 flex items-center">
                <FileText className="w-3 h-3 mr-2" /> Record Data Summary
              </h4>
              <div className="space-y-2">
                 {Object.entries(record.details).map(([key, value]) => (
                   key !== 'name' && key !== 'institution' && key !== 'branch' && (
                     <div key={key} className="flex justify-between text-sm border-b border-slate-200/50 dark:border-slate-700/50 pb-1 last:border-0">
                       <span className="text-slate-500 capitalize">{key.replace(/([A-Z])/g, ' $1').trim()}</span>
                       <span className="font-medium text-slate-700 dark:text-slate-300">{String(value)}</span>
                     </div>
                   )
                 ))}
                 <div className="flex justify-between text-sm pt-2">
                    <span className="text-slate-500">Status</span>
                    <span className="font-bold text-emerald-600">{record.status}</span>
                 </div>
              </div>
           </div>
        </CardContent>

        <CardFooter className="bg-slate-50 dark:bg-slate-800/80 border-t border-slate-100 dark:border-slate-800 p-4 flex justify-between gap-3">
          <Button variant="outline" size="sm" onClick={handleExportJson} className="text-xs">
            <Copy className="w-3 h-3 mr-2" /> JSON
          </Button>
          <div className="flex space-x-3">
             <Button variant="secondary" size="sm" onClick={() => onLoadAsDraft(record)} className="text-xs">
                <RefreshCw className="w-3 h-3 mr-2" /> LOAD AS DRAFT
             </Button>
             <Button size="sm" onClick={handleRegeneratePdf} disabled={isRegenerating} className="text-xs bg-slate-800 text-white hover:bg-slate-700">
                {isRegenerating ? <RefreshCw className="w-3 h-3 mr-2 animate-spin" /> : <Download className="w-3 h-3 mr-2" />}
                REGENERATE PDF
             </Button>
          </div>
        </CardFooter>
      </Card>
    </div>
  );
};
