import React, { useEffect, useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/Dialog';
import { Button } from '@/components/ui/Button';
import { Loader2, CheckCircle, FileCheck, ShieldCheck, Download } from 'lucide-react';
import { Progress } from '@/components/ui/Progress';

interface PdfGenerationModalProps {
    isOpen: boolean;
    onClose: () => void;
    onComplete: () => void; // Trigger actual download
    studentName: string;
}

type Stage = 'INIT' | 'COMPILING' | 'SIGNING' | 'READY';

export function PdfGenerationModal({ isOpen, onClose, onComplete, studentName }: PdfGenerationModalProps) {
    const [stage, setStage] = useState<Stage>('INIT');
    const [progress, setProgress] = useState(0);

    useEffect(() => {
        if (isOpen && stage === 'INIT') {
            startGeneration();
        }
        if (!isOpen) {
            // Reset on close
            setStage('INIT');
            setProgress(0);
        }
    }, [isOpen]);

    const startGeneration = () => {
        setStage('COMPILING');
        setProgress(20);

        setTimeout(() => {
            setStage('SIGNING');
            setProgress(70);
        }, 2000);

        setTimeout(() => {
            setStage('READY');
            setProgress(100);
        }, 4000);
    };

    const handleDownload = () => {
        onComplete();
        onClose();
    };

    return (
        <Dialog open={isOpen} onOpenChange={(open) => !open && onClose()}>
            <DialogContent className="sm:max-w-md">
                <DialogHeader>
                    <DialogTitle>Generating Official Academic Record</DialogTitle>
                </DialogHeader>
                
                <div className="py-6 flex flex-col items-center justify-center space-y-6">
                    {stage !== 'READY' ? (
                        <div className="relative">
                            <div className="absolute inset-0 bg-indigo-500/20 blur-xl rounded-full"></div>
                            <Loader2 className="h-16 w-16 text-indigo-600 animate-spin relative z-10" />
                        </div>
                    ) : (
                        <div className="relative">
                             <div className="absolute inset-0 bg-green-500/20 blur-xl rounded-full"></div>
                             <FileCheck className="h-16 w-16 text-green-600 relative z-10" />
                        </div>
                    )}

                    <div className="w-full space-y-2">
                        <div className="flex justify-between text-xs font-mono text-slate-500 uppercase">
                            <span>Status</span>
                            <span>{stage === 'READY' ? 'COMPLETE' : `${progress}%`}</span>
                        </div>
                        <Progress value={progress} className="h-2" />
                    </div>

                    <div className="space-y-3 w-full">
                        <StageItem 
                            label="Retrieving SIS Data & Compiling Record" 
                            status={progress >= 20 ? (progress > 20 ? 'done' : 'active') : 'pending'} 
                        />
                         <StageItem 
                            label="Applying Digital Certifications & Seals" 
                            status={progress >= 70 ? (progress > 70 ? 'done' : 'active') : 'pending'} 
                        />
                    </div>
                </div>

                <DialogFooter className="sm:justify-center">
                    {stage === 'READY' ? (
                        <Button onClick={handleDownload} className="w-full bg-green-600 hover:bg-green-700">
                            <Download className="w-4 h-4 mr-2" /> Download Packet PDF
                        </Button>
                    ) : (
                        <Button variant="ghost" disabled className="w-full text-slate-400">
                            Processing...
                        </Button>
                    )}
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}

function StageItem({ label, status }: { label: string, status: 'pending' | 'active' | 'done' }) {
    return (
        <div className={`flex items-center space-x-3 text-sm transition-all duration-300 ${status === 'pending' ? 'opacity-40' : 'opacity-100'}`}>
            <div className={`w-5 h-5 flex items-center justify-center rounded-full border ${
                status === 'done' ? 'bg-green-100 border-green-200 text-green-600' : 
                status === 'active' ? 'bg-indigo-100 border-indigo-200 text-indigo-600 animate-pulse' : 
                'bg-slate-50 border-slate-200 text-slate-300'
            }`}>
                {status === 'done' ? <CheckCircle className="w-3 h-3" /> : 
                 status === 'active' ? <Loader2 className="w-3 h-3 animate-spin" /> : 
                 <div className="w-1.5 h-1.5 rounded-full bg-current" />}
            </div>
            <span className={status === 'active' ? 'font-bold text-indigo-700' : 'text-slate-600'}>
                {label}
            </span>
        </div>
    );
}
