import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/Card';
import { PacketRecord } from '@/types/student-verification';
import { FileText, Copy, Trash2, Eye, RefreshCw } from 'lucide-react'; // Assuming lucide-react is available or use similar icons

interface PacketHistoryPanelProps {
  history: PacketRecord[];
  onView: (record: PacketRecord) => void;
  onDuplicate: (record: PacketRecord) => void;
  onDelete: (id: string) => void;
}

export const PacketHistoryPanel: React.FC<PacketHistoryPanelProps> = ({
  history,
  onView,
  onDuplicate,
  onDelete
}) => {
  if (history.length === 0) {
    return (
      <Card className="border-slate-800 bg-slate-900/50">
        <CardHeader>
          <CardTitle className="text-sm font-medium text-slate-400">GENERATED PACKETS HISTORY</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center py-8 text-slate-500 text-sm">
            No packets generated in this session history.
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className="border-slate-800 bg-slate-900/50">
      <CardHeader className="flex flex-row items-center justify-between">
        <CardTitle className="text-sm font-medium text-purple-400">GENERATED PACKETS HISTORY</CardTitle>
        <span className="text-xs text-slate-500">{history.length} RECORDS STORED</span>
      </CardHeader>
      <CardContent className="p-0">
        <div className="overflow-x-auto">
          <table className="w-full text-sm text-left">
            <thead className="text-xs text-slate-400 uppercase bg-slate-900/80 border-b border-slate-800">
              <tr>
                <th className="px-4 py-3">System Ref</th>
                <th className="px-4 py-3">Student</th>
                <th className="px-4 py-3">Institution</th>
                <th className="px-4 py-3">Term</th>
                <th className="px-4 py-3">Generated At</th>
                <th className="px-4 py-3 text-right">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-slate-800">
              {history.map((record) => (
                <tr key={record.id} className="hover:bg-slate-800/50 transition-colors">
                  <td className="px-4 py-3 font-mono text-xs text-purple-300">
                    {record.systemRef.split(' ').slice(2).join(' ')} {/* Show just the unique part if possible, or full */}
                  </td>
                  <td className="px-4 py-3 font-medium text-slate-200">{record.studentName}</td>
                  <td className="px-4 py-3 text-slate-400 truncate max-w-[150px]">{record.institution}</td>
                  <td className="px-4 py-3 text-slate-400">{record.term}</td>
                  <td className="px-4 py-3 text-slate-500 text-xs">
                    {new Date(record.generatedAt).toLocaleString()}
                  </td>
                  <td className="px-4 py-3 text-right space-x-2">
                    <button 
                      onClick={() => onView(record)}
                      className="text-slate-400 hover:text-blue-400 transition-colors p-1"
                      title="View Details"
                    >
                      <Eye size={16} />
                    </button>
                    <button 
                      onClick={() => onDuplicate(record)}
                      className="text-slate-400 hover:text-green-400 transition-colors p-1"
                      title="Duplicate / Load Data"
                    >
                      <Copy size={16} />
                    </button>
                    <button 
                      onClick={() => onDelete(record.id)}
                      className="text-slate-400 hover:text-red-400 transition-colors p-1"
                      title="Delete Record"
                    >
                      <Trash2 size={16} />
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </CardContent>
    </Card>
  );
};
