"use client";

import React, { useEffect, useState } from 'react';
import { Select } from './ui/Select';
import { MONTHS } from '@/lib/constants';

interface DateSelectProps {
  value?: Date | null;
  onChange: (date: Date | null) => void;
  startYear?: number;
  endYear?: number;
  disabled?: boolean;
}

export function DateSelect({ 
  value, 
  onChange, 
  startYear = 1950, 
  endYear = new Date().getFullYear(),
  disabled 
}: DateSelectProps) {
  const [selectedMonth, setSelectedMonth] = useState<string>('');
  const [selectedDay, setSelectedDay] = useState<string>('');
  const [selectedYear, setSelectedYear] = useState<string>('');

  // Sync internal state with external value
  useEffect(() => {
    if (value) {
      setSelectedMonth(value.getMonth().toString());
      setSelectedDay(value.getDate().toString());
      setSelectedYear(value.getFullYear().toString());
    } else {
      // Don't clear if the user is in the middle of selecting...
      // Actually for "Autofill" we need this sync.
      // For manual clearing, we also need it.
      // But we shouldn't overwrite if value is undefined and we have local state? 
      // Simplified: Layout usually controls this.
      if (value === null) {
          setSelectedMonth('');
          setSelectedDay('');
          setSelectedYear('');
      }
    }
  }, [value]);

  const years = Array.from({ length: endYear - startYear + 1 }, (_, i) => endYear - i);
  const daysInMonth = selectedMonth && selectedYear 
    ? new Date(Number.parseInt(selectedYear), Number.parseInt(selectedMonth) + 1, 0).getDate() 
    : 31;

  const days = Array.from({ length: daysInMonth }, (_, i) => i + 1);

  const handleChange = (type: 'month' | 'day' | 'year', val: string) => {
    let newMonth = selectedMonth;
    let newDay = selectedDay;
    let newYear = selectedYear;

    if (type === 'month') {
      setSelectedMonth(val);
      newMonth = val;
    } else if (type === 'day') {
      setSelectedDay(val);
      newDay = val;
    } else if (type === 'year') {
      setSelectedYear(val);
      newYear = val;
    }

    if (newMonth !== '' && newDay !== '' && newYear !== '') {
      const date = new Date(Number.parseInt(newYear), Number.parseInt(newMonth), Number.parseInt(newDay));
      // Basic validation to ensure the date is valid (e.g. not Feb 30)
      if (date.getMonth() === Number.parseInt(newMonth)) {
         onChange(date);
      } else {
         // Invalid date (e.g. Feb 31), maybe reset day?
         // For now let's just update local state and let the effect sync or wait for valid.
         // Actually, if date is invalid, Date object might roll over.
         // Let's constrain the day if it exceeds max.
         const maxDay = new Date(Number.parseInt(newYear), Number.parseInt(newMonth) + 1, 0).getDate();
         if (Number.parseInt(newDay) > maxDay) {
            // Adjust day
             const validDate = new Date(Number.parseInt(newYear), Number.parseInt(newMonth), maxDay);
             onChange(validDate);
             setSelectedDay(maxDay.toString()); 
         } else {
             onChange(date);
         }
      }
    } else {
      onChange(null);
    }
  };

  return (
    <div className="grid grid-cols-3 gap-2 sm:gap-4">
      <Select 
        value={selectedMonth} 
        onChange={(e) => handleChange('month', e.target.value)}
        disabled={disabled}
        aria-label="Select Month"
      >
        <option value="" disabled>Month</option>
        {MONTHS.map((month, index) => (
          <option key={month} value={index}>
            {month}
          </option>
        ))}
      </Select>
      
      <Select 
        value={selectedDay} 
        onChange={(e) => handleChange('day', e.target.value)}
        disabled={disabled}
        aria-label="Select Day"
      >
        <option value="" disabled>Day</option>
        {days.map((day) => (
          <option key={day} value={day}>
            {day}
          </option>
        ))}
      </Select>

      <Select 
        value={selectedYear} 
        onChange={(e) => handleChange('year', e.target.value)}
        disabled={disabled}
        aria-label="Select Year"
      >
        <option value="" disabled>Year</option>
        {years.map((year) => (
          <option key={year} value={year}>
            {year}
          </option>
        ))}
      </Select>
    </div>
  );
}
