import React, { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/Dialog';
import { Button } from '@/components/ui/Button';
import { Label } from '@/components/ui/Label';
import { AlertCircle, CheckCircle, FileText, HelpCircle } from 'lucide-react';
import { Course } from '@/types/student-verification';

interface BulkCourseImporterProps {
  isOpen: boolean;
  onClose: () => void;
  onImport: (courses: Course[]) => void;
}

export function BulkCourseImporter({ isOpen, onClose, onImport }: BulkCourseImporterProps) {
  const [inputText, setInputText] = useState('');
  const [parsedCourses, setParsedCourses] = useState<Course[]>([]);
  const [error, setError] = useState<string | null>(null);

  const handleParse = () => {
    setError(null);
    if (!inputText.trim()) {
      setError('Please paste some data first.');
      return;
    }

    const lines = inputText.split('\n').filter(line => line.trim().length > 0);
    const courses: Course[] = [];

    lines.forEach((line, index) => {
      // Try to intelligently split by tab or common delimiters if tab fails
      // Priority: Tab -> Pipe -> Semicolon -> Comma (risky for titles)
      let parts = line.split('\t');
      
      // Fallback strategies for common copy-pastes
      if (parts.length < 2) parts = line.split('|');
      if (parts.length < 2) parts = line.split(';');
      
      // If still failed, maybe it's just spaces? No, that's too dangerous.
      // Let's stick to structured delimiters.

      // Map parts to Course fields. Expecting order: Code, Title, Instructor, Days, Time, Room, Credits
      // We'll try to be flexible.
      
      // Basic mapping (Code, Title, Credits) are essential.
      const code = parts[0]?.trim() || `CRS-${index + 1}`;
      const title = parts[1]?.trim() || 'Untitled Course';
      const instructor = parts[2]?.trim() || '';
      const days = parts[3]?.trim() || '';
      const time = parts[4]?.trim() || '';
      const room = parts[5]?.trim() || '';
      const creditsStr = parts[6]?.trim() || parts[2]?.trim(); // Fallback if someone pastes Code | Title | Credits
      
      const credits = parseInt(creditsStr) || 3;

      courses.push({
        id: Math.random().toString(36).substr(2, 9),
        code,
        title,
        instructor,
        days,
        time,
        room,
        credits
      });
    });

    if (courses.length === 0) {
      setError('Could not parse any courses. Ensure data is separated by tabs or pipes.');
    } else {
      setParsedCourses(courses);
    }
  };

  const handleConfirm = () => {
    onImport(parsedCourses);
    setParsedCourses([]);
    setInputText('');
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[600px]">
        <DialogHeader>
          <DialogTitle>Bulk Import Courses</DialogTitle>
        </DialogHeader>
        
        <div className="space-y-4 py-2">
          <div className="text-sm text-slate-500 bg-slate-50 p-3 rounded border border-slate-200">
            <p className="font-semibold flex items-center mb-1">
              <HelpCircle className="w-3 h-3 mr-1" /> Instructions
            </p>
            <p>Paste your course data below. Supported format (Tab or Pipe separated):</p>
            <code className="block mt-2 bg-white p-1 rounded text-xs border">
              CODE | TITLE | INSTRUCTOR | DAYS | TIME | ROOM | CREDITS
            </code>
            <p className="mt-1 text-xs">Example: CS101 | Intro to CS | Dr. Smith | MWF | 10:00 | A101 | 3</p>
          </div>

          <div className="space-y-2">
            <Label>Raw Data</Label>
            <textarea
              className="w-full h-40 p-3 text-xs font-mono border rounded-md focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
              placeholder={`CS101\tIntro to CS\tDr. Smith\tMWF\t10:00\tA101\t3\nMATH200\tCalculus II\tProf. Doe\tTTh\t13:00\tB202\t4`}
              value={inputText}
              onChange={(e) => setInputText(e.target.value)}
            />
          </div>

          <div className="flex justify-end">
            <Button variant="outline" size="sm" onClick={handleParse}>
              Preview Parse
            </Button>
          </div>

          {error && (
            <div className="flex items-center text-red-600 text-sm bg-red-50 p-2 rounded">
              <AlertCircle className="w-4 h-4 mr-2" />
              {error}
            </div>
          )}

          {parsedCourses.length > 0 && (
            <div className="space-y-2">
              <Label className="flex items-center text-green-700">
                <CheckCircle className="w-4 h-4 mr-2" />
                Valid Courses Found ({parsedCourses.length})
              </Label>
              <div className="max-h-40 overflow-y-auto border rounded divide-y text-xs">
                {parsedCourses.map((c, i) => (
                  <div key={i} className="p-2 grid grid-cols-12 gap-2 hover:bg-slate-50">
                    <div className="col-span-2 font-bold">{c.code}</div>
                    <div className="col-span-6 truncate">{c.title}</div>
                    <div className="col-span-2 text-slate-500">{c.credits}cr</div>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={onClose}>Cancel</Button>
          <Button onClick={handleConfirm} disabled={parsedCourses.length === 0}>
            Import {parsedCourses.length} Courses
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
