import { NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
// import { ServiceRecord as ServiceRecordType } from '@/types/records'; // Unused

const ensureTableExists = async () => {
    // No-op: handled by Prisma migrations
};

export async function GET() {
  try {
    const records = await prisma.serviceRecord.findMany({
      orderBy: {
        createdAt: 'desc',
      },
      take: 100,
    });
    
    // Map Prisma model back to ServiceRecord type
    // The 'snapshot' field holds the full JSON we want.
    // Or we might need to map specific fields if they were extracted.
    // Our schema has 'snapshot' as the full payload.
    const mappedRecords = records.map(r => r.snapshot);

    return NextResponse.json({ ok: true, records: mappedRecords });
  } catch (error: any) {
    console.error('Failed to fetch records:', error);
    return NextResponse.json(
      { ok: false, error: 'Failed to fetch records' },
      { status: 500 }
    );
  }
}

export async function POST(request: Request) {
  try {
    const body = await request.json();
    const { record } = body;

    if (!record || !record.id) {
        return NextResponse.json({ ok: false, error: 'Invalid record data' }, { status: 400 });
    }

    // Upsert using Prisma
    await prisma.serviceRecord.upsert({
      where: { systemRef: record.systemRef || record.id }, // Using systemRef as unique key if available, else fallback logic needed. 
      // Schema has systemRef as unique. 
      // ServiceRecord type usually has systemRef.
      // let's check ServiceRecord type in useServiceRecords or assume systemRef exists.
      // If record.systemRef is missing, we might have issues.
      // But typically it exists for generated packets.
      create: {
        id: record.id,
        type: record.type,
        systemRef: record.systemRef,
        title: record.title || 'Untitled',
        subjectName: record.details?.name || 'Unknown',
        institution: record.details?.institution,
        term: record.details?.term,
        includedDocs: record.includedDocs || [], // Assuming it's in record
        snapshot: record,
      },
      update: {
        title: record.title || 'Untitled',
        subjectName: record.details?.name || 'Unknown',
        institution: record.details?.institution,
        term: record.details?.term,
        includedDocs: record.includedDocs || [],
        snapshot: record,
      }
    });

    return NextResponse.json({ ok: true });
  } catch (error: any) {
    console.error('Failed to save record:', error);
    return NextResponse.json(
      { ok: false, error: 'Failed to save record', details: error.message },
      { status: 500 }
    );
  }
}

export async function DELETE(request: Request) {
    try {
        const { searchParams } = new URL(request.url);
        const id = searchParams.get('id');

        if (!id) {
            return NextResponse.json({ ok: false, error: 'Missing ID' }, { status: 400 });
        }
        
        await prisma.serviceRecord.delete({
            where: { id },
        });

        return NextResponse.json({ ok: true });
    } catch (error: any) {
        console.error('Failed to delete record:', error);
        return NextResponse.json({ ok: false, error: 'Failed to delete record' }, { status: 500 });
    }
}
