import { NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import { headers } from 'next/headers';

export async function GET() {
  // 1. Security Check
  const headersList = await headers(); // Await headers() in Next.js 15/16
  const token = headersList.get('x-health-token');
  const isDev = process.env.NODE_ENV !== 'production';

  // In production, require the correct token
  if (!isDev && token !== process.env.HEALTHCHECK_TOKEN) {
    return NextResponse.json(
      { ok: false, error: 'Unauthorized' },
      { status: 401 }
    );
  }

  try {
    // 2. Connection Check
    // Perform a simple query to assert connection and permissions
    const start = Date.now();
    const result: any[] = await prisma.$queryRaw`SELECT current_database() as db, current_user as "user", version() as version`;
    const duration = Date.now() - start;
    
    const row = result[0];

    return NextResponse.json({
      ok: true,
      timestamp: new Date().toISOString(),
      duration: `${duration}ms`,
      meta: {
        database: row.db,
        user: row.user,
        version: row.version,
      }
    });
  } catch (error: any) {
    console.error('Database Health Check Failed:', error);
    
    // 3. Error Diagnostics (Safe)
    // Extract PG connection details safely for debugging without exposing password
    const dbUrl = process.env.DATABASE_URL || '';
    let connectionMeta = 'Not set';
    
    try {
        if (dbUrl.startsWith('postgres')) {
            const url = new URL(dbUrl);
            connectionMeta = `${url.protocol}//${url.username}:***@${url.hostname}:${url.port}${url.pathname}`;
        }
    } catch (e) {
        connectionMeta = 'Invalid URL format';
    }

    return NextResponse.json(
      { 
        ok: false, 
        error: 'Database connection failed',
        code: error.code, // SQLState if available
        meta: {
            attempted_connection: connectionMeta,
            env_ssl: process.env.DB_SSL,
            message: error.message
        }
      },
      { status: 500 }
    );
  }
}

