/*
 * Database Bootstrap Script
 * -------------------------
 * This script is idempotent. It will:
 * 1. Create the 'genekei_user' role if it doesn't exist.
 * 2. Create the 'genekei' database if it doesn't exist.
 * 3. Grant necessary privileges.
 * 
 * Usage:
 *   psql -U postgres -f scripts/db/bootstrap.sql
 *   (Ensure you have a superuser connection or sufficient privileges)
 */

-- 1. Create Role (genekei_user)
DO
$do$
BEGIN
   IF NOT EXISTS (
      SELECT FROM pg_catalog.pg_roles
      WHERE  rolname = 'genekei_user') THEN

      CREATE ROLE genekei_user WITH LOGIN PASSWORD 'genekeipass';
      RAISE NOTICE 'Role "genekei_user" created. PLEASE CHANGE THE PASSWORD IMMEDIATELY IF THIS IS PRODUCTION.';
   ELSE
      RAISE NOTICE 'Role "genekei_user" already exists. Skipping creation.';
   END IF;
END
$do$;

-- 2. Create Database (genekei)
-- Note: 'CREATE DATABASE' cannot run inside a DO block or transaction block.
-- We use a widely compatible workaround or rely on the user running this conditionally.
-- However, for a single script, we can check existence via \gexec in psql or just fail nicely.
-- Since this runs in various tools, the safest SQL-standard way is hard without \gexec.
-- We will just attempt to creating it and ignore error if exists isn't easily possible in standard SQL script without external logic.
-- BUT, we can use a trick:
SELECT 'CREATE DATABASE genekei OWNER genekei_user'
WHERE NOT EXISTS (SELECT FROM pg_database WHERE datname = 'genekei')\gexec

-- 3. Grant Privileges
-- Connect to the new database to grant schema privileges.
-- NOTE: In a single script run via psql, we can't switch DBs easily midway without \c.
-- WE ASSUME this script is run, and then if DB didn't exist, it's created.
-- The user might need to run the grants separately if they aren't superuser or if they are in a specific tool.
-- However, we can grant privileges on the DATABASE itself from here.

GRANT ALL PRIVILEGES ON DATABASE genekei TO genekei_user;

-- End of Bootstrap
